
/* ---- Inline Script #1 (extracted) ---- */

(function patchFetch() {
    const orig = window.fetch;
    window.fetch = (url, opts = {}) => {
        opts.credentials = "include";
        opts.headers = Object.assign({}, opts.headers || {});
        const sid = localStorage.getItem("rscpid_sid");
        if (sid) opts.headers["Authorization"] = "Bearer " + sid;
        return orig(url, opts);
    };
})();


let AUTO_INLINE = false;
let inlineEl = null;

function inlineStart(prefixText = '', prefixClass = 'fg', rowClass = 'small') {
  const log = document.getElementById('log');
  if (!log) return;
  const row = document.createElement('div');
  row.className = rowClass;
  const pre = document.createElement('span');
  pre.textContent = prefixText;
  pre.className = prefixClass; 
  const dyn = document.createElement('span');
  dyn.id = 'inline-txt';
  row.appendChild(pre);
  row.appendChild(dyn);
  log.appendChild(row);
  inlineEl = dyn;
  log.scrollTop = log.scrollHeight;
}


function inlineSet(text, cssClass = 'muted') {
  if (!inlineEl) return;
  inlineEl.className = cssClass;
  inlineEl.textContent = text;
  const log = document.getElementById('log');
  if (log) log.scrollTop = log.scrollHeight;
}

function inlineEnd(addNewLine = true) {
  inlineEl = null;
  if (addNewLine) {
    const log = document.getElementById('log');
    if (log) {
      const br = document.createElement('div');
      br.className = 'small';
      br.textContent = '';
      log.appendChild(br);
      log.scrollTop = log.scrollHeight;
    }
  }
}

let CN_WAIT_T = null;   
let CN_START_TO = null;  
let CN_SEND_T = null;     
let CN_CANCEL = false;   
const CN_TZ = 'Asia/Shanghai';


function setInlineStatus(text, css = 'muted') {
  const log = $('log');
  if (!log) return;
  let node = document.getElementById('inline-cn-status');
  if (!node) {
    node = document.createElement('div');
    node.id = 'inline-cn-status';
    node.className = css;
    log.appendChild(node);
  }
  node.className = css;
  node.textContent = text;
  log.scrollTop = log.scrollHeight;
}

function chinaNow() {

  return new Date(new Date().toLocaleString('en-US', { timeZone: CN_TZ }));
}

function makeChinaDateLike(nowC, h, m, s) {
  const d = new Date(nowC); 
  d.setHours(h, m, s, 0);
  return d;
}

function determineChinaWindow(nowC) {
  const mins = nowC.getHours() * 60 + nowC.getMinutes();
  const m00 = 0;
  const m01 = 60;
  const m2358 = 23 * 60 + 58;
  const m0028 = 24 * 60 + 28;

  let startC, endC;
  if (mins < m01) {
    startC = nowC;
    endC = makeChinaDateLike(nowC, 1, 0, 0);
  } else if (mins < m2358) {
    startC = makeChinaDateLike(nowC, 23, 58, 0);
    endC = makeChinaDateLike(nowC, 0, 28, 0);
  } else if (mins < m0028) {
    startC = nowC;
    endC = makeChinaDateLike(nowC, 0, 28, 0);
  } else {
    const tomorrow = new Date(nowC);
    tomorrow.setDate(tomorrow.getDate() + 1);
    startC = makeChinaDateLike(tomorrow, 23, 58, 0);
    endC = makeChinaDateLike(tomorrow, 0, 28, 0);
  }
  return { startC, endC };
}


function clearChinaTimers() {
  if (CN_WAIT_T) { clearInterval(CN_WAIT_T); CN_WAIT_T = null; }
  if (CN_START_TO) { clearTimeout(CN_START_TO); CN_START_TO = null; }
  if (CN_SEND_T) { clearInterval(CN_SEND_T); CN_SEND_T = null; }
}

function stopChinaAuto(reason = 'İptal') {
  CN_CANCEL = true;
  clearChinaTimers();
  setInlineStatus(`China Auto: ${reason}.`, 'warn');
  $('chinaAutoHint') && ($('chinaAutoHint').textContent = '');
}

async function startChinaAuto() {
  CN_CANCEL = false;
  clearChinaTimers();

  const dl = document.getElementById('downloadLink');
  if (dl) dl.style.display = 'none';

  const nowC = chinaNow();
  const { startC, endC } = determineChinaWindow(nowC);

  if (chinaNow() < startC) {
    logLine(`China Time bekleniyor…`, 'info');
    const hint = document.getElementById('chinaAutoHint');
    if (hint) hint.textContent = `Başlangıç: ${startC.toTimeString().slice(0,8)} (CN)`;

    CN_WAIT_T = setInterval(() => {
      const remain = Math.max(0, startC - chinaNow());
      const hh = String(Math.floor(remain / 3600000)).padStart(2, '0');
      const mm = String(Math.floor((remain % 3600000) / 60000)).padStart(2, '0');
      const ss = String(Math.floor((remain % 60000) / 1000)).padStart(2, '0');
      setInlineStatus(`China wait → ${hh}:${mm}:${ss}`, 'muted');
    }, 1000);

    CN_START_TO = setTimeout(() => {
      if (CN_CANCEL) return;
      if (CN_WAIT_T) { clearInterval(CN_WAIT_T); CN_WAIT_T = null; }
      runChinaTryLoop(startC /* startAt */);
    }, Math.max(0, startC - chinaNow()));
  } else {
    const hint = document.getElementById('chinaAutoHint');
    if (hint) hint.textContent = `Pencere aktif (CN)`;
    runChinaTryLoop(chinaNow() /* startAt */);
  }
}


function runChinaTryLoop(startAt) {
  AUTO_INLINE = true;   
  inlineStart('Send Imei Server : ', 'fg'); 

  const endLocal = new Date(startAt.getTime() + 30 * 60 * 1000);

  CN_SEND_T = setInterval(async () => {
    if (CN_CANCEL) {
      inlineSet('Canceled', 'warn');
      inlineEnd(true);
      AUTO_INLINE = false;
      clearChinaTimers();
      return;
    }

    const now = chinaNow();
    if (now >= endLocal) {
      inlineSet('Ended (no success)', 'warn');
      inlineEnd(true);
      AUTO_INLINE = false;
      clearChinaTimers();
      return;
    }

    const remain = Math.max(0, endLocal - now);
    const hh = String(Math.floor(remain / 3600000)).padStart(2, '0');
    const mm = String(Math.floor((remain % 3600000) / 60000)).padStart(2, '0');
    const ss = String(Math.floor((remain % 60000) / 1000)).padStart(2, '0');

    inlineSet(`Trying…`, 'warn');

    await createCriticalOnServer(); 

    const dl = document.getElementById('downloadLink');
    const ok = dl && dl.style.display !== 'none';
    if (ok) {
      inlineSet('✔️', 'ok');
      inlineEnd(true);
      AUTO_INLINE = false;
      clearChinaTimers();
      return;
    } else {
      inlineSet(`Fail (Auto 30min kalan: ${hh}:${mm}:${ss})`, 'err');
    }
  }, 1000);
}




let currentUserCredit = 0;
const FORCE_UNLIMITED_CREDIT = false;
const AUTH_SYSTEM_DISABLED = false;
const CREDIT_UNLIMITED_LABEL = "∞ (sınırsız)";
const CREDIT_UNLIMITED_VALUE = Number.MAX_SAFE_INTEGER;
const AUTH_ALLOWED_ORIGINS = [
  "https://rscpidpanel.pages.dev",
  "http://localhost",
  "https://localhost",
  "http://127.0.0.1",
  "https://127.0.0.1"
];

function isAuthOriginAllowed(origin) {
  if (!origin || origin === "null") return true;
  return AUTH_ALLOWED_ORIGINS.some((allowed) => origin === allowed || origin.startsWith(allowed + ":"));
}

function formatCreditLabel(value) {
  if (FORCE_UNLIMITED_CREDIT) return CREDIT_UNLIMITED_LABEL;
  const num = Number(value);
  if (!Number.isFinite(num)) return "0";
  return String(num);
}

function setCurrentUserCredit(nextValue) {
  currentUserCredit = FORCE_UNLIMITED_CREDIT
    ? Number.POSITIVE_INFINITY
    : (Number.isFinite(nextValue) ? nextValue : 0);
  const el = $("userCredit");
  if (el) el.textContent = "credit: " + formatCreditLabel(currentUserCredit);
  if (typeof updateButtonStates === "function") updateButtonStates();
}
const AUTH_BASE = "https://rscpid-auth.gmrec72.workers.dev";
const API_BASE = AUTH_BASE + "/proxy/";
const SECRET = "RSCpid2024";
const XOR_KEY = "X98Q2YU9UD9839D93UJD93UJD9832JUD9J23DUJ2938JDU9238UJD9238JD9283JD9283JD2938DJ9238JD9238DJ2398DJ2398DJ293JD928JD9283DUJU9283JD2938JD2938J";

const CountryNames = {
  GLOBAL:"GLOBAL", GLOBAL_DC:"GLOBAL", MI:"GLOBAL", XM:"GLOBAL", EEA:"EUROPE (EEA)",
  CN:"CHINA", IN:"INDIA", ID:"INDONESIA", VN:"VIETNAM", TH:"THAILAND", BN:"BANGLADESH",
  PK:"PAKISTAN", NP:"NEPAL", LK:"SRI LANKA", PH:"PHILIPPINES", MY:"MALAYSIA",
  SG:"SINGAPORE", TW:"TAIWAN", KR:"SOUTH KOREA", JP:"JAPAN", HK:"HONG KONG",
  TR:"TURKEY", IQ:"IRAQ", EG:"EGYPT", DZ:"ALGERIA", MA:"MOROCCO", TN:"TUNISIA",
  LY:"LIBYA", SA:"SAUDI ARABIA", AE:"UNITED ARAB EMIRATES", IL:"ISRAEL", JO:"JORDAN",
  KW:"KUWAIT", QA:"QATAR", OM:"OMAN", YE:"YEMEN", RU:"RUSSIA", UA:"UKRAINE",
  BY:"BELARUS", KZ:"KAZAKHSTAN", UZ:"UZBEKISTAN", KG:"KYRGYZSTAN", AM:"ARMENIA",
  GE:"GEORGIA", AZ:"AZERBAIJAN", MX:"MEXICO", BR:"BRAZIL", AR:"ARGENTINA",
  CL:"CHILE", CO:"COLOMBIA", PE:"PERU", ZA:"SOUTH AFRICA", NG:"NIGERIA",
  GH:"GHANA", KE:"KENYA", UNKNOWN:"UNKNOWN REGION"
};

const $ = (id) => document.getElementById(id);
let adbDevice = null;
let isConnected = false;
let isProcessing = false;
let fastbootTransport = null;
let fastbootDevice = null;
let fbTransport = null;
let fbDevice = null;
let isFastbootConnected = false;
let adr_masterAppList = [];
let inputs = {
  imei1: $('imei1'), imei2: $('imei2'), imei1Luhn: $('imei1Luhn'), imei2Luhn: $('imei2Luhn'),
  wmac: $('wmac'), btmac: $('btmac'), product: $('product'), cpuId: $('cpuId'),
  psntoimeitxt: $('psntoimeitxt'), csntoimeitxt: $('csntoimeitxt')
};

function formatUpper(v){ return (v||'').toString().trim().toUpperCase(); }

function hexToGB(hex){
  if(!hex) return null;
  let h = hex.trim();
  if(/^0x/i.test(h)) h = h.slice(2);
  let dec = 0n;
  try{ dec = BigInt('0x'+h); } catch { return null; }
  const gb = Number(dec) / (1024*1024*1024);
  return gb;
}

function closestPhoneSizeGB(gb){
  const sizes = [8,16,32,64,128,256,512,1024];
  let best = sizes[0], diff = Infinity;
  for(const s of sizes){ const d=Math.abs(s-gb); if(d<diff){diff=d; best=s;} }
  return best;
}

async function fbGetVar(name){
  try{
    await fbDevice.send(`getvar:${name}`);
    const resp = await fbDevice.receive();
    const cmd  = fbDevice.get_cmd(resp);
    if(cmd==="OKAY"||cmd==="INFO"){
      const payload = new TextDecoder().decode(fbDevice.get_payload(resp)).trim();
      return payload;
    }
  }catch(_){}
  return "";
}


let LOG_LINE_SPACING = 1;
function setLogLineSpacing(px){ LOG_LINE_SPACING = px; }

function logLine(html, className, fontSize = null) {
  const logBox = $("log");
  const s = document.createElement("div");
  s.innerHTML = html;
  if (className) s.classList.add(className);

  s.style.margin = `${LOG_LINE_SPACING}px 0`;
  s.style.padding = "0";
  s.style.lineHeight = "1.25";
  s.style.display = "block";

	const defaultFontSize = 15; 

	if (fontSize) {
	  const sizeVal = isNaN(fontSize) ? fontSize : fontSize + "px";
	  s.style.fontSize = sizeVal;
	} else {
	  s.style.fontSize = defaultFontSize + "px";
	}


  logBox.appendChild(s);
  logBox.scrollTop = logBox.scrollHeight;
}

function logTwoColor(label, value, labelCls = "muted", valueCls = "info", fontSize = null, sep = " ") {
  let baseStyle = "";
  const defaultFontSize = 14;
  const sizeVal = isNaN(fontSize) ? fontSize : (fontSize ? fontSize + "px" : defaultFontSize + "px");
  baseStyle = ` style="font-size:${sizeVal}"`;

  const colorMap = {
    red:   'var(--err)',
    green: 'var(--ok)',
    blue:  'var(--primary)',
    orange:'var(--amber)'
  };

  const isColorToken = (t) =>
    !!t && (
      t in colorMap ||
      /^#([0-9a-f]{3}|[0-9a-f]{6}|[0-9a-f]{8})$/i.test(t) ||
      /^rgb|^hsl/i.test(t)
    );

  const labelIsColor = isColorToken(labelCls);
  const valueIsColor = isColorToken(valueCls);

  const labelStyle = labelIsColor
    ? baseStyle.slice(0, -1) + `; color:${colorMap[labelCls] || labelCls}"`
    : baseStyle;
  const valueStyle = valueIsColor
    ? baseStyle.slice(0, -1) + `; color:${colorMap[valueCls] || valueCls}"`
    : baseStyle;

  const labelClassAttr = labelIsColor ? '' : ` class="${labelCls}"`;
  const valueClassAttr = valueIsColor ? '' : ` class="${valueCls}"`;

  const sepHtml = `<span style="font-size:6px;">${sep}</span>`;

  logLine(
    `<span${labelClassAttr}${labelStyle}>${label}</span>${sepHtml}` +
    `<span${valueClassAttr}${valueStyle}>${value}</span>`,
    null,
    fontSize
  );
}


function updateGlobalStatus(text, color = "orange") { const globalEl = $('global-action-text'); if (globalEl) { globalEl.textContent = text; let cssColor = 'var(--amber)'; if (color === 'green' || color === '#22c55e') cssColor = 'var(--ok)'; else if (color === 'red' || color === '#ef4444') cssColor = 'var(--err)'; else if (color === 'blue' || color === '#3B82F6') cssColor = 'var(--primary)'; globalEl.style.color = cssColor; } }



function updateStatus(text, progress = null) {
  const s = $('status-text'); if (s) s.textContent = text;
  if (progress !== null) {
    const p = $('progress-bar-fill');
    if (p) p.style.width = `${progress}%`;
  }
}

function updateGlobalStatus(text, color = "orange") {
  const globalEl = $('global-action-text');
  if (globalEl) {
    globalEl.textContent = text;
    let cssColor = 'var(--amber)';
    if (color === 'green' || color === '#22c55e') cssColor = 'var(--ok)';
    else if (color === 'red' || color === '#ef4444') cssColor = 'var(--err)';
    else if (color === 'blue' || color === '#3B82F6') cssColor = 'var(--primary)';
    globalEl.style.color = cssColor;
  }
}

function sanitize14(v) {
    return v.replace(/\D/g, "").slice(0, 14);
}

function calcLuhn14to15(n14) {
    if (typeof n14 !== 'string' || n14.length !== 14) return '';
    let sum = 0, dbl = true;
    for (let i = n14.length - 1; i >= 0; i--) {
        let d = n14.charCodeAt(i) - 48;
        if (dbl) {
            d = d * 2;
            if (d > 9) d -= 9;
        }
        sum += d;
        dbl = !dbl;
    }
    return String((10 - (sum % 10)) % 10);
}

function obfuscate(jsonStr) {
    const enc = new TextEncoder().encode(jsonStr);
    const key = new TextEncoder().encode(XOR_KEY);
    const out = new Uint8Array(enc.length);
    for (let i = 0; i < enc.length; i++) out[i] = enc[i] ^ key[i % key.length];
    const b64 = btoa(String.fromCharCode(...out));
    return (b64 + SECRET).split("").reverse().join("");
}

function updateButtonStates() {
  const hasEnoughCredit = currentUserCredit >= 10;
  const isAdbConnected = isConnected;

  document.querySelectorAll('#readInfoAdbButton, [data-adb]')
    .forEach(btn => btn.disabled = !isAdbConnected);

  const fastbootBtns = ['fbReadInfoButton','fbSwitchSlotButton','fbRebootSystemButton'];
  fastbootBtns.forEach(id => {
    const b = $(id);
    if (b) b.disabled = false; 
  });

  document.querySelectorAll('#tab-sn-query-content .btn, #tab-sn-query-content .input')
    .forEach(el => { el.disabled = !hasEnoughCredit; });

  const adsRemoverActive = hasEnoughCredit && isAdbConnected;
  document.querySelectorAll('#tab-ads-remover-content .btn, #tab-ads-remover-content .input')
    .forEach(el => el.disabled = !adsRemoverActive);
}



function forceDisconnect(silent = false) {
    document.querySelectorAll('[data-adb], #readInfoAdbButton, #adr-analiz, #adr-system-cleaner, #adr-onerilen, #adr-etkinlestir, #adr-devredisi, #adr-sil, #adr-uygulamaara')
        .forEach(el => el.disabled = true);

    if (adbDevice) { adbDevice = null; }
    isConnected = false;

    const btn = $('connectButton');
    if (btn) {
        btn.innerHTML = '<i data-feather="smartphone" style="width:20px; height:20px; margin-right: 8px;"></i> Cihaza Bağlan';
        feather.replace(); 
        btn.disabled = false;
    }

    updateButtonStates();

    if (silent) {
        updateStatus('Cihaz bağlantısı bekleniyor...', 0);
        updateGlobalStatus('İşlem Bekleniyor...', 'amber');
    } else {
        updateStatus('Bağlantı Kaybedildi. Yeniden bağlanın.', 0);
        updateGlobalStatus('Bağlantı Kaybedildi', 'red');
    }
}




async function connectDevice() {
  if (isConnected) { logLine('Cihaz zaten ADB ile bağlı.', "ok"); return; }
  if (isFastbootConnected) { logLine('Cihaz zaten Fastboot ile bağlı.', "warn"); updateGlobalStatus('Cihaz Bağlı (Fastboot)', 'blue'); updateButtonStates(); return; }

  logLine('Cihaza ADB üzerinden bağlanılıyor...', "info");
  updateStatus('Cihaz Seçimi Açılıyor...', 10);

  try {
    if (typeof Adb === 'undefined' || typeof Adb.open !== 'function')
      throw new Error('webadb.js yüklü değil');

    const transport = await Adb.open('WebUSB');

    try {
            adbDevice = await transport.connectAdb("host::");
            isConnected = true;
            logLine('Bağlantı Başarılı! Cihaz kullanıma hazır (ADB).', "ok");
            updateStatus('Cihaz Bağlı', 100);

            const connectBtn = $('connectButton');
            connectBtn.innerHTML = '<i data-feather="smartphone" style="width:20px; height:20px; margin-right: 8px;"></i> Cihaz Bağlı';
            feather.replace(); 
            connectBtn.disabled = true;

            updateButtonStates();
            updateGlobalStatus('Cihaz Bağlı (ADB)', 'green');

    } catch (adbErr) {
      try {
        const fbTransport = await Adb.open('WebUSB');
        const fb = await fbTransport.connectFastboot();
        if (fb) {
          fastbootTransport = fbTransport;
          fastbootDevice = fb;
          isFastbootConnected = true;
          logLine('Fastboot modunda cihaz bulundu !', 'warn');
          updateGlobalStatus('Cihaz Bağlı (Fastboot)', 'blue');
          updateButtonStates();
        } else {
          try { await fbTransport.close(); } catch {}
          throw new Error('Fastboot bulunamadı.');
        }
      } catch (fbErr) {
        try { await transport.close(); } catch {}
        throw fbErr || adbErr;
      }
    }
  } catch (e) {
    forceDisconnect();
    logLine(`Hata: Cihaz bağlantısı kurulamadı! ${e.message}`, "err");
    updateStatus('Bağlantı Başarısız', 0);
    updateGlobalStatus('Bağlantı Başarısız', 'red');
  }
}

async function ensureFastboot(){
  if (fbDevice) return true;

  try{
    if(!fbTransport) fbTransport = await Adb.open('WebUSB');
    fbDevice = await fbTransport.connectFastboot();
    return !!fbDevice;
  }catch(e){
    fbDevice = null;
    try{ if(fbTransport) await fbTransport.close(); }catch{}
    fbTransport = null;
    return false;
  }
}

async function closeFastbootIfNeeded(){
  try{ if(fbTransport){ await fbTransport.close(); } }catch{}
  fbDevice=null; fbTransport=null;
}

async function readInfoFastbootPro(){
  if(isProcessing) return;
  isProcessing = true;
  logLine('\n--- Fastboot Bilgi Okuma Başladı ---', 'info');
  updateGlobalStatus("Fastboot Cihazı Aranıyor...", "blue");

  try{
    const ok = await ensureFastboot();
    if(!ok) throw new Error("Fastboot cihazı seçilmedi / bağlanamadı.");

    // C# karşılıkları
    const dp          = await fbGetVar('DP');                         // vendor’a göre bazen olur
    const serial      = await fbGetVar('serialno');
    const product     = await fbGetVar('product');
    const unlocked    = await fbGetVar('unlocked');
    const batt        = await fbGetVar('battery-voltage');
    const token       = await fbGetVar('token');
    const secure      = await fbGetVar('secure');
    const variant     = await fbGetVar('variant');                    // Memory Type gibi
    const anti        = await fbGetVar('anti');
    const board_ver   = await fbGetVar('board_version');
    const curSlot     = await fbGetVar('current-slot');
    const baseband    = await fbGetVar('version-baseband');
    const partUserHex = await fbGetVar('partition-size:userdata');    // 0x…
    const cpuidRaw    = await fbGetVar('cpuid');
    const chipid      = await fbGetVar('chipid');

    const isMtk = (baseband||'').toLowerCase().includes('moly');
    logTwoColor("Chipset type", isMtk ? "MEDIATEK" : "QUALCOMM", "muted", isMtk ? "warn" : "ok");

    if (dp)         logTwoColor("DP", formatUpper(dp), "muted", "ok");
    if (serial)     logTwoColor("Serial", formatUpper(serial), "muted", "info");
    if (product){
      const prodU = formatUpper(product);
      logTwoColor("Product", prodU, "muted", "warn");
      inputs.product.value = prodU.toLowerCase();
    }
    if (anti)       logTwoColor("Anti ver", formatUpper(anti), "muted", "warn");
    if (board_ver)  logTwoColor("Board ver", formatUpper(board_ver), "muted", "info");

    if (curSlot){
      const slot = /b/i.test(curSlot) ? "B" : "A";
      logTwoColor("Current Slot", slot, "muted", "info");
    }

    if (batt)       logTwoColor("Battery volt", formatUpper(batt), "muted", "warn");
    if (secure)     logTwoColor("Is boot secure", formatUpper(secure), "muted", "warn");
    if (variant)    logTwoColor("Memory Type", formatUpper(variant), "muted", "warn");
    if (baseband)   logTwoColor("Baseband", formatUpper(baseband), "muted", "info");

    let cpuid = (cpuidRaw||'').trim();
    if (cpuid && !/^0x/i.test(cpuid)) cpuid = '0x' + cpuid;
    if (cpuid){
      logTwoColor("Cpuid", formatUpper(cpuid), "muted", "ok");
      inputs.cpuId.value = cpuid.toLowerCase();
    }
    if (chipid)     logTwoColor("Chipid", formatUpper(chipid), "muted", "ok");

    if (partUserHex){
      const gb = hexToGB(partUserHex);
      if (gb !== null){
        const approx = closestPhoneSizeGB(gb);
        logTwoColor("Total Storage Size", `(${approx} GB)`, "muted", "ok");
        logTwoColor("Available Storage Size", `(${gb.toFixed(2)} GB)`, "muted", "ok");
      }
    }

    const isUnlocked = /yes/i.test(unlocked||'');
    logTwoColor("Bootloader", isUnlocked ? "UNLOCKED" : "LOCKED", "muted", isUnlocked ? "ok" : "err");


    inputs.wmac.value = randomizeMac(""); 
    inputs.btmac.value = randomizeMac("");
    checkAllInputs(); 

    updateGlobalStatus("Bilgi Okuma Tamamlandı", "green");
  }catch(e){
    logLine(`Fastboot hatası: ${e.message}`, 'err');
    updateGlobalStatus("İşlem Başarısız", "red");
  }finally{
    isProcessing=false;
    logLine('--- İşlem Sonu ---', 'info');

  }
}


async function adbShellCommand(cmd, timeoutMs = 20000) {
    if (!adbDevice) {
        logLine('ADB Cihazı bağlı değil!', 'err');
        return null;
    };
    try {
        const stream = await adbDevice.shell(cmd);
        const decoder = new TextDecoder();
        let output = "";
        const deadline = Date.now() + timeoutMs;

        while (true) {
            const timeout = Math.max(0, deadline - Date.now());
            if (timeout === 0) {
                try { await stream.close(); } catch {}
                return output.trim();
            }
            const msg = await Promise.race([
                stream.receive(),
                new Promise((_, rej) => setTimeout(() => rej(new Error("shell-timeout")), timeout))
            ]);
            if (msg.cmd === "WRTE") {
                if (msg.data && msg.data.byteLength) {
                    output += decoder.decode(msg.data);
                }
                await stream.send("OKAY");
                continue;
            }
            if (msg.cmd === "OKAY") continue;
            if (msg.cmd === "CLSE") {
                await stream.close();
                break;
            }
            if (msg.cmd === "FAIL") throw new Error("ADB stream FAIL");
            break;
        }
        return output.trim();
    } catch (e) {
        if ((e.message || "").toLowerCase().includes("disconnected")) {
            forceDisconnect();
            return null;
        }
        console.error("adbShellCommand Hatası:", e);
        logLine(`Komut hatası (${cmd}): ${e.message}`, 'err');
        return null;
    }
}

async function processFastbootResponse(fastboot, command) {
    await fastboot.send(command);
    const response = await fastboot.receive();
    const cmd = fastboot.get_cmd(response);
    if (cmd === "OKAY" || cmd === "INFO") {
        const payload = new TextDecoder().decode(fastboot.get_payload(response));
        return payload.trim();
    }
    return null;
}

async function fastbootRebootSystem(){
  if (isProcessing) return;
  isProcessing = true;
  logLine('\n--- Fastboot Reboot (System) ---', 'info');
  updateGlobalStatus('Cihaz yeniden başlatılıyor...', 'blue');

  try{
    const ok = await ensureFastboot();
    if (!ok) throw new Error('Fastboot cihazı bağlı değil.');

    await fbDevice.send('reboot');
    logLine('Fastboot: reboot komutu gönderildi.', 'ok');
    updateStatus('Yeniden başlatılıyor...', 100);
    updateGlobalStatus('Reboot gönderildi', 'green');

    try { await closeFastbootIfNeeded(); } catch {}
    isFastbootConnected = false;
    forceDisconnect(true);
  }catch(e){
    logLine(`Fastboot reboot hatası: ${e.message}`, 'err');
    updateGlobalStatus('İşlem Başarısız', 'red');
  }finally{
    isProcessing = false;
    logLine('--- İşlem Sonu ---', 'info');
  }
}


async function switchSlotFastboot() {
    if (isProcessing) return;
    isProcessing = true;
    logLine('\n--- Fastboot Slot Değiştirme Başladı ---', 'info');
    updateGlobalStatus("Fastboot Cihazı Aranıyor...", "blue");

    try {
        const isConnected = await ensureFastboot(); 
        if (!isConnected) {
            throw new Error("Fastboot cihazına bağlanılamadı.");
        }

        updateGlobalStatus("Slot Bilgisi Alınıyor...", "blue");
        
        const currentSlot = await processFastbootResponse(fbDevice, "getvar:current-slot");
        if (!currentSlot || currentSlot === "null" || currentSlot === "") {
            throw new Error("Cihaz A/B slot sistemini desteklemiyor veya slot bilgisi alınamadı.");
        }
        logTwoColor("Mevcut Slot:", currentSlot, "muted", "warn");

        const nextSlot = currentSlot === 'a' ? 'b' : 'a';
        updateGlobalStatus(`Slot ${nextSlot} olarak değiştiriliyor...`, "blue");

        const switchResponse = await processFastbootResponse(fbDevice, `set_active:${nextSlot}`);
        logTwoColor("Slot Değiştirme Sonucu:", switchResponse || "OK", "muted", "ok");
        logTwoColor("Yeni Aktif Slot:", nextSlot.toUpperCase(), "muted", "ok");

        updateGlobalStatus("Cihaz Yeniden Başlatılıyor...", "blue");
        logLine("Cihaz yeniden başlatılıyor...", "warn");
        await fbDevice.send("reboot");
        updateGlobalStatus("İşlem Tamamlandı", "green");
    } catch (e) {
        logLine(`Fastboot hatası: ${e.message}`, 'err');
        updateGlobalStatus("İşlem Başarısız", "red");
    } finally {
        isProcessing = false;
        logLine('--- İşlem Sonu ---', 'info');
    }
}


function ToHexCpuid(input) {
    if (!input) return '';
    input = input.trim();
    if (input.startsWith('0x') || input.startsWith('0X')) input = input.substring(2);
    if (/^\d+$/.test(input)) { try { return BigInt(input).toString(16); } catch (_) { return ''; } }
    if (/^[0-9a-fA-F]+$/.test(input)) return input.toLowerCase();
    
    const idx = input.indexOf(':');
    if(idx >=0 && idx+1 < input.length) input = input.substring(idx+1).trim();
    input = input.replace(/[^0-9a-fA-F]/g,'').replace(/^0+/, '');
    if(input.length === 0) return '0';
    return input;
}

function ExtractPropertyValue(source, identifier) {
    if (!source) return null;
    const regex = new RegExp(`\\[${identifier}\\]: \\[([^\\]]+)\\]`);
    const match = source.match(regex);
    return match ? match[1] : null;
}

function isLikelyRealUid(hx) {
  if (!hx) return false;
  let h = hx.trim().toLowerCase();
  if (h.startsWith('0x')) h = h.slice(2);
  if (!/^[0-9a-f]+$/.test(h)) return false;
  if (!(h.length >= 16 && h.length <= 32)) return false;
  const DUMMIES = ['deadbeef','deadc0de','decafbad','badc0de','badf00d','feedface','cafebabe','baadf00d','0','ffff'];
  if (DUMMIES.some(d => h.includes(d))) return false;
  if (/be+e|de+ad|de+ed|c0de|c0ea|cec/.test(h)) return h.length > 20; // çok kısaysa at
  return true;
}

function normalizeUid(hx) {
  let h = (hx || '').trim().toLowerCase();
  if (h.startsWith('0x')) h = h.slice(2);
  h = h.replace(/[^0-9a-f]/g, '').replace(/^0+/, '');
  if (h.length === 0) return '';
  if (h.length < 16) h = h.padStart(16, '0');
  if (h.length > 16 && h.length < 32) h = h.padStart(32, '0');
  return '0x' + h;
}


	function _lc(s){ return (s||"").toLowerCase().trim(); }

	async function _rootName() {
	  const v1 = await adbShellCommand('su -v');
	  const v2 = await adbShellCommand('su -V');
	  const part1 = (v1 || '').trim();
	  const part2 = (v2 || '').trim();
	  return [part1, part2].filter(Boolean).join(' ');
	}

	async function detectRoot() {
	  const out = await adbShellCommand('su -c id', 8000);
	  const low = _lc(out);

	  if (low.includes('uid=0') || low.includes('(root)') || low.includes('gid=0')) {
		const name = await _rootName();
		return { rooted: true, method: 'su -c id', name };
	  }

	  if (low.includes('permission not obtained') ||
		  low.includes('permission denied') ||
		  low.includes('not found') ||
		  low.includes('no such file')) {
		const which = await adbShellCommand('which su', 4000);
		const hasSu = !!which && !_lc(which).includes('not found') && which.trim() !== '';
		return { rooted: false, hasSu, reason: 'no-permission-or-not-found' };
	  }

	  const whichSu = await adbShellCommand('which su', 4000);
	  const hasSu = !!whichSu && !_lc(whichSu).includes('not found') && whichSu.trim() !== '';
	  const buildTags = await adbShellCommand('getprop ro.build.tags', 3000);
	  const isTestKeys = _lc(buildTags).includes('test-keys');

	  return { rooted: false, hasSu, isTestKeys };
	}


async function readInfoLikeCSharp(silent = false, fillForm = true) {
    if (!isConnected) {
        logLine('Hata: Cihaza bağlı değilsiniz. Lütfen önce bağlanın.', 'err');
        return;
    }
    if (!silent) logLine('\n--- ADB Bilgi Okuma Başladı ---', 'info');
    isProcessing = true;
    updateGlobalStatus("Cihaz Bilgileri Okunuyor...", "blue");

    try {
        const run = async (cmd, msg, progress) => {
            if (!isConnected) throw new Error("Bağlantı kesildi");
            updateStatus(msg, progress);
            const output = await adbShellCommand(cmd);
            if (output === null && !isConnected) throw new Error("Bağlantı kesildi");
            await new Promise(r => setTimeout(r, 50));
            return output;
        };

        const getpropOutput = await run('getprop', 'Prop bilgileri alınıyor...', 20);
        if (!getpropOutput) throw new Error("Temel getprop bilgileri okunamadı.");

        const memInfo = await run('cat /proc/meminfo | grep MemTotal', 'RAM bilgisi alınıyor.', 40);
        const imeiOutput = await run('service call iphonesubinfo 1 s16 com.android.shell | cut -c 52-66 | tr -d ".[:space:]"', 'IMEI bilgisi alınıyor.', 60);
        const macBtOutput = await run('settings get secure bluetooth_address', 'MAC bilgisi alınıyor.', 75);
        const cpuidOutput = await run('cat /sys/devices/soc0/serial_number', 'CPU ID okunuyor.', 90);

        if (!silent) logLine("--- Cihaz Bilgileri ---", "primary");
        
        const product = ExtractPropertyValue(getpropOutput, 'ro.product.device') || '';
        if (product) {
            logTwoColor("Product:", product.toUpperCase(), "muted", "ok");
            if (fillForm) inputs.product.value = product.toLowerCase();
        }

        const brand = ExtractPropertyValue(getpropOutput, 'ro.product.manufacturer') || ExtractPropertyValue(getpropOutput, 'ro.product.product.brand') || '';
        if(brand) logTwoColor("Marka:", brand.toUpperCase(), "muted", "warn");

        const model = ExtractPropertyValue(getpropOutput, 'ro.product.model') || '';
        if(model) logTwoColor("Model:", model.toUpperCase(), "muted", "info");

        const marketname = ExtractPropertyValue(getpropOutput, 'ro.product.marketname') || '';
        if(marketname) logTwoColor("Model Adı:", marketname.toUpperCase(), "muted", "ok");
		
		
        const androidVer = ExtractPropertyValue(getpropOutput, 'ro.build.version.release') || ExtractPropertyValue(getpropOutput, 'ro.system.build.version.release') || '';
        if(androidVer) logTwoColor("Android Versiyon:", androidVer, "muted", "info");

        const miuiVer = ExtractPropertyValue(getpropOutput, 'ro.build.version.incremental') || '';
        if(miuiVer) logTwoColor("MIUI Versiyon:", miuiVer, "muted", "info", "14");

       
		let chosenRam = null;

		if (memInfo) {
			const m = /MemTotal:\s*(\d+)\s*kB/i.exec(memInfo);
			if (m) {
				const ramGB = Math.round(parseInt(m[1], 10) / (1024 * 1000));
				const standard = [2, 3, 4, 6, 8, 12, 16, 24, 32];
				chosenRam = standard.find(s => ramGB <= s) || ramGB;
			}
		}

		const processor =
		  ExtractPropertyValue(getpropOutput, 'ro.boot.hardware') ||
		  ExtractPropertyValue(getpropOutput, 'ro.board.platform') || '';

		if (processor) {
			const ramText = chosenRam ? ` (${chosenRam} GB RAM)` : '';
			logTwoColor("İşlemci:", `${processor.toUpperCase()}${ramText}`, "muted", "info");
		}



        const imei1 = ExtractPropertyValue(getpropOutput, 'ro.ril.miui.imei0') || ExtractPropertyValue(getpropOutput, 'ro.ril.oem.imei1') || imeiOutput || '';
        if (imei1 && imei1.length >= 15) {
            logTwoColor("IMEI 1:", imei1, "muted", "warn");
            if (fillForm) inputs.imei1.value = imei1.substring(0, 14);
        }
        
        const imei2 = ExtractPropertyValue(getpropOutput, 'ro.ril.miui.imei1') || ExtractPropertyValue(getpropOutput, 'ro.ril.oem.imei2') || '';
        if (imei2 && imei2.length >= 15) {
            logTwoColor("IMEI 2:", imei2, "muted", "warn");
            if (fillForm) inputs.imei2.value = imei2.substring(0, 14);
        }
         
		 
		const hwInfo    = ExtractPropertyValue(getpropOutput, 'ro.boot.hwc') || '';
		const hwVersion = ExtractPropertyValue(getpropOutput, 'ro.boot.hwversion') || '';
		const hwLevel   = ExtractPropertyValue(getpropOutput, 'ro.boot.hwlevel') || '';

		if (hwInfo || hwVersion || hwLevel) {
		  const lvlClass = hwLevel.toUpperCase() === 'MP' ? 'ok' : 'err';

		  const parts = [];

		  parts.push(`<span class="muted">Hw Info:</span>`);


		  let first = true;
		  const addVal = (cls, v) => {
			if (!v) return;
			if (!first) parts.push(`<span class="muted">/</span>`); 
			parts.push(`<span class="${cls}">${v.toUpperCase()}</span>`);
			first = false;
		  };

		  addVal('info', hwInfo);
		  addVal('warn', hwVersion);
		  addVal(lvlClass, hwLevel);

		  logLine(parts.join(''));
		}
		 
        const baseband = ExtractPropertyValue(getpropOutput, 'gsm.version.baseband') || '';
        if(baseband) logTwoColor("Baseband:", baseband, "muted", "orange", "13");

        const psn = ExtractPropertyValue(getpropOutput, 'ro.ril.oem.psno') || '';
        if(psn) {
            logTwoColor("PSN:", psn, "muted", "info", "13");
            if(fillForm) inputs.psntoimeitxt.value = psn;
        }

        const csn = ExtractPropertyValue(getpropOutput, 'ro.ril.oem.sno') || '';
        if(csn) {
            logTwoColor("CSN:", csn, "muted", "info", "13");
            if(fillForm) inputs.csntoimeitxt.value = csn;
        }
        
		
		const blRaw =
		  (ExtractPropertyValue(getpropOutput, 'ro.secureboot.lockstate') ||
		   ExtractPropertyValue(getpropOutput, 'ro.boot.vbmeta.device_state') ||
		   ExtractPropertyValue(getpropOutput, 'ro.boot.flash.locked') || 
		   ExtractPropertyValue(getpropOutput, 'ro.boot.verifiedbootstate') || 
		   '').toLowerCase().trim();

		let blText = '';
		let blClass = 'info';

		
		if (blRaw) {
		  if (blRaw === 'unlocked' || blRaw === "red" || blRaw === '0') {
			blText = 'UNLOCKED';
			blClass = 'ok';    
		  } else if (blRaw === 'locked' || blRaw === 'green' || blRaw === '1') {
			blText = 'LOCKED';
			blClass = "red"; 
		  } else {
			blText = blRaw.toUpperCase();
			blClass = 'info';
		  }

		  logTwoColor('Bootloader:', blText, 'muted', blClass);
		}


		const chipidProp = ToHexCpuid(ExtractPropertyValue(getpropOutput, 'ro.boot.chipid'));
		const cpuidProp  = ToHexCpuid(ExtractPropertyValue(getpropOutput, 'ro.boot.cpuid'));
		const hwchipid   = ToHexCpuid(ExtractPropertyValue(getpropOutput, 'ro.boot.hwchipid'));
		const socSerial  = ToHexCpuid(cpuidOutput);

		const ordered = [chipidProp, cpuidProp, hwchipid, socSerial].filter(Boolean);

		const candidates = ordered.filter(isLikelyRealUid);

		let finalCpuid = '';
		if (candidates.length > 0) {
		  finalCpuid = normalizeUid(candidates[0]);
		} else {
		  const fallback = ordered.find(Boolean) || '';
		  finalCpuid = normalizeUid(fallback);
		}

		if (finalCpuid) {
		  logTwoColor("CPU ID:", finalCpuid, "muted", "ok");
		  if (fillForm) inputs.cpuId.value = finalCpuid;
		} else {
		  logTwoColor("CPU ID:", "N/A", "muted", "err");
		  if (fillForm) inputs.cpuId.value = "N/A";
		}


        const macBt = macBtOutput ? macBtOutput.trim().replace(/[^0-9a-fA-F]/g, '').toUpperCase() : '';
        if (macBt && macBt.length === 12) {
            const last = macBt.slice(-1);
            const nextHex = ((parseInt(last, 16) + 1) % 16).toString(16).toUpperCase();
            const wmac = macBt.slice(0, -1) + nextHex;
            logTwoColor("BT MAC:", macBt, "muted", "warn", "13");
            logTwoColor("WiFi MAC:", wmac, "muted", "warn", "13");
            if (fillForm) {
                inputs.btmac.value = macBt;
                inputs.wmac.value = wmac;
            }
        }
        
		const root = await detectRoot();

		if (root.rooted) {
		  const extra = root.name ? ` {${root.name}}` : '';
		  logTwoColor("Root:", `EVET${extra}`, "muted", "ok");
		} else {
		  if (root.hasSu) {
			logTwoColor("Root:", "HAYIR (su mevcut ama erişim yok)", "muted", "red");
		  } else {
			logTwoColor("Root:", "HAYIR", "muted", "red");
		  }
		  if (root.isTestKeys) {
			logTwoColor("Build Tags:", "test-keys (debug yapı)", "muted", "red");
		  }
		}
		
		
        if(fillForm) checkImeiLuhn();
        updateStatus('Tüm bilgiler okundu.', 100);
        updateGlobalStatus("Bilgi Okuma Tamamlandı", "green");

    } catch (e) {
        logLine(`Bilgi okuma hatası: ${e.message || e}`, 'err');
        updateStatus('Bilgi okuma hatası', 0);
        updateGlobalStatus("İşlem Başarısız", "red");
    } finally {
        isProcessing = false;
        if (!silent) logLine('--- İşlem Sonu ---', 'info');
    }
}

let adr_allItems = [];
const adr_iconCache = new Map();
const ADR_DEFAULT_EXCLUDES = new Set([
  "app.revanced.android.gms",
  "app.revanced.android.youtube",
  "ar.tvplayer.tv",
  "biz.mobinex.android.apps.cep_sifrematik",
  "cn.wps.moffice_eng",
  "cn.wps.xiaomi.abroad.lite",
  "com.abtnprojects.ambatana",
  "com.adobe.lrmobile",
  "com.ahoygames.okey",
  "com.akbank.android.apps.akbank_direkt",
  "com.android.soundrecorder",
  "com.androxus.batterymeter",
  "com.aktifbank.nkolay",
  "com.azure.authenticator",
  "com.binance.dev",
  "com.building.craft.everyday.loki.fun.cliffs.builder",
  "com.cloud.tr",
  "com.denizbank.mobildeniz",
  "com.discord",
  "com.dolap.android",
  "com.duokan.phone.remotecontroller",
  "com.facebook.katana",
  "com.facebook.lite",
  "com.facebook.orca",
  "com.finansbank.mobile.cepsube",
  "finansbank.enpara",
  "com.ForgeGames.SpecialForcesGroup2",
  "com.gamma.scan",
  "com.gardrops",
  "com.garanti.cepsubesi",
  "com.google.android.apps.adm",
  "com.google.android.apps.chromecast.app",
  "com.google.android.apps.docs",
  "com.google.android.apps.nbu.files",
  "com.google.android.apps.photos",
  "com.google.android.apps.tachyon",
  "com.google.android.apps.videos",
  "com.google.android.ims",
  "com.google.android.music",
  "com.google.android.play.games",
  "com.govee.home",
  "com.haremaltin.android.haremaltin",
  "com.spotify.music",
  "com.inomera.sm",
  "com.instagram.android",
  "com.instagram.lite",
  "com.keenetic.kn",
  "com.kiloo.subwaysurf",
  "com.king.candycrushsaga",
  "com.kokteyl.mackolik",
  "com.kuveytturk.mobil",
  "com.lgeha.nuts",
  "com.lmr.lfm",
  "com.longcheertel.midtest",
  "com.lydsto.robotvacuum",
  "com.mb.android",
  "com.mercusys.halo",
  "com.mi.global.bbs",
  "com.mi.global.shop",
  "com.mi.globalbrowser",
  "com.micredit.in",
  "com.miui.android.fashiongallery",
  "com.miui.calculator",
  "com.miui.compass",
  "com.miui.mediaeditor",
  "com.miui.notes",
  "com.miui.screenrecorder",
  "com.miui.weather",
  "com.miui.weather2",
  "com.mobillium.papara",
  "com.mobilexsoft.ezanvakti",
  "com.multicraft.game",
  "com.nekki.shadowfight3",
  "com.netflix.mediaclient",
  "com.niksoftware.snapseed",
  "com.oceanwing.battery.cam",
  "com.pozitron.hepsiburada",
  "com.pozitron.iscep",
  "com.preff.kb.xm",
  "com.reqable.android",
  "com.revanced.net.revancedmanager",
  "com.riotgames.mobile.leagueconnect",
  "com.sahibinden",
  "com.sec.android.app.sbrowser",
  "com.shazam.android",
  "com.snapchat.android",
  "com.solidict.gnc2",
  "com.teamviewer.quicksupport.market",
  "com.teamviewer.teamviewer.market.mobile",
  "com.think.earth",
  "com.tmob.AveaOIM",
  "com.tplink.iot",
  "com.tplink.tether",
  "com.ttech.android.onlineislem",
  "com.turkcell.bip",
  "com.turkcell.paycell",
  "com.twitter.android",
  "com.vakifbank.mobile",
  "com.valvesoftware.android.steam.community",
  "com.valvesoftware.android.steam.friendsui",
  "com.vodafone.selfservis",
  "com.whatsapp",
  "com.whatsapp.w4b",
  "com.xiaomi.midpop",
  "com.xiaomi.midrop",
  "com.xiaomi.scanner",
  "com.xiaomi.smarthome",
  "com.xiaozhen.beauty.shenglong",
  "com.yablio.sendfilestotv",
  "com.yandex.browser",
  "com.ykb.android",
  "com.ykb.avm",
  "com.zhiliaoapp.musically",
  "com.zhiliaoapp.musically.go",
  "com.ziraat.basakdijital",
  "com.ziraat.ziraatmobil",
  "com.ziraatkatilim.mobilebanking",
  "io.github.dovecoteescapee.byedpi",
  "net.peakgames.Yuzbir",
  "net.zedge.android",
  "org.codeaurora.snapcam",
  "org.jellyfin.mobile",
  "org.telegram.messenger",
  "tr.gov.saglik.ekipportal",
  "tr.gov.saglik.enabiz",
  "tr.gov.saglik.MHRSMOBIL",
  "tr.gov.turkiye.edevlet.kapisi",
  "trendyol.com",
  "com.trendyol.go"
]);
const ADR_FORCE_INCLUDE = new Set([
  "app.phonedoctor.cleaner",
  "app1352864.mdq",
  "co.shopcats.shopcats",
  "com.aminesk.statussaver",
  "com.block.juggle",
  "com.block.puzzle.game.hippo.mi",
  "com.blockpuzzle.comboblastmaster.freegame",
  "com.bng.calculator",
  "com.crazy.juicer.xm",
  "com.eggs.bloodflow",
  "com.gcmforex.pro",
  "com.glamour.superpapers",
  "com.hotflyiohl",
  "com.jewelsblast.ivygames.Adventure.free",
  "com.logame.eliminateintruder3d",
  "com.mintgames.triplecrush.tile.fun",
  "com.mintgames.wordtrip",
  "com.mirror.rabbits.fill",
  "com.miui.cleanmaster",
  "com.miui.hybrid",
  "com.mxxtech.easyscan",
  "com.muslimappsforyou.qiblafinder",
  "com.mysuruhawking.flutter_grocery",
  "com.nf.snake",
  "com.pdeoniros.sioepocdq",
  "com.pop.star.popstar.funny",
  "com.popstar.luckypuzzle.dmmc",
  "com.powerplay.backgrounds",
  "com.precisedefend.vitiaz",
  "com.samsung.sree",
  "com.scanner.ploks",
  "com.scenic.vistaspaper",
  "com.sgakagak.agakagabs",
  "com.simpledi.cleanmy",
  "com.splashismail.gamefruitss",
  "com.starmakerinteractive.starmaker",
  "com.sukhavati.gotoplaying.bubble.BubbleShooter.mint",
  "com.suosderam.drapeisoe",
  "com.tb.mobiclean",
  "com.theme.sleepsound.abcd",
  "com.tidyupclean.cleanup.tool",
  "com.tranquil.scenes",
  "com.xiaomi.mipicks",
  "com.xisueoezj.eoeuiroll",
  "com.zelony.car.wallpaper",
  "com.zelony.mercedes.wallpaper",
  "com.zrayga.firstapp",
  "m4emt8no.s86y8bhi.ahtad",
  "reasoning.reasoning.reasoning.reasoning",
  "speed.sc.cleaner",
  "zipper.ziplocker.zipperlockscreen.zipwallpaper.ziplock"
]);

const ADR_BAD_WORDS = ["clean","boost","battery","locker","antivirus","remove","remover","ads","scan","optimizer","vpn","proxy","speed","booster","free","pro","plus","tool"];
const ADR_AD_PROVIDERS = ["admob","unity3d.ads","mbridge","bigo.ads","vungle","applovin","chartboost","ironsource","facebook.ads","startapp"];
const ADR_ANALYTICS  = ["flurry","firebase.analytics","appsflyer","adjust","mixpanel"];


async function systemCleaner() {
    if (isProcessing || !isConnected) {
        logLine("Zaten bir işlem devam ediyor veya cihaz bağlı değil.", "warn");
        return;
    }
    isProcessing = true;
    updateGlobalStatus('Sistem Temizleyici Başladı...', 'blue');
    logLine('\n--- Sistem Temizleyici Başladı ---', 'info');

const packages = [
    "app1352864.mdq",
    "co.shopcats.shopcats",
    "com.aminesk.statussaver",
    "com.block.juggle",
    "com.block.puzzle.game.hippo.mi",
    "com.blockpuzzle.comboblastmaster.freegame",
    "com.bng.calculator",
    "com.bsp.catchlog",
    "com.crazy.juicer.xm",
    "com.eggs.bloodflow",
    "com.gcmforex.pro",
    "com.glamour.superpapers",
    "com.hotflyiohl",
    "com.jewelsblast.ivygames.Adventure.free",
    "com.lmr.lfm",
    "com.logame.eliminateintruder3d",
    "com.mi.global.shop",
    "com.mi.globalbrowser",
    "com.mipay.wallet.id",
    "com.mipay.wallet.in",
    "com.mintgames.triplecrush.tile.fun",
    "com.mintgames.wordtrip",
    "com.miui.analytics",
    "com.miui.android.fashiongallery",
    "com.miui.bugreport",
    "com.miui.cleanmaster",
    "com.miui.hybrid",
    "com.miui.hybrid.accessory",
    "com.miui.msa.global",
    "com.muslimappsforyou.qiblafinder",
    "com.mysuruhawking.flutter_grocery",
    "com.mxxtech.easyscan",
    "com.netflix.mediaclient",
    "com.netflix.partner.activation",
    "com.nf.snake",
    "com.opera.app.news",
    "com.opera.branding",
    "com.opera.branding.news",
    "com.opera.mini.native",
    "com.pdeoniros.sioepocdq",
    "com.pop.star.popstar.funny",
    "com.popstar.luckypuzzle.dmmc",
    "com.powerplay.backgrounds",
    "com.precisedefend.vitiaz",
    "com.samsung.sree",
    "com.scanner.ploks",
    "com.scenic.vistaspaper",
    "com.sgakagak.agakagabs",
    "com.simpledi.cleanmy",
    "com.splashismail.gamefruitss",
    "com.starmakerinteractive.starmaker",
    "com.sukhavati.gotoplaying.bubble.BubbleShooter.mint",
    "com.suosderam.drapeisoe",
    "com.tb.mobiclean",
    "com.theme.sleepsound.abcd",
    "com.tidyupclean.cleanup.tool",
    "com.tranquil.scenes",
    "com.xiaomi.joyose",
    "com.xiaomi.mipicks",
    "com.xiaomi.simactivate.service",
    "com.xisueoezj.eoeuiroll",
    "com.zelony.car.wallpaper",
    "com.zelony.mercedes.wallpaper",
    "com.zrayga.firstapp",
    "in.amazon.mShop.android.shopping",
    "in.mohalla.sharechat",
    "m4emt8no.s86y8bhi.ahtad",
    "reasoning.reasoning.reasoning.reasoning",
    "speed.sc.cleaner",
    "zipper.ziplocker.zipperlockscreen.zipwallpaper.ziplock"
];
    
    try {
        logLine("Reklam ve gereksiz sistem paketleri kaldırılıyor...", 'warn');
        for (let i = 0; i < packages.length; i++) {
            const pkg = packages[i];
            const progress = Math.round(((i + 1) / packages.length) * 100);
            updateStatus(`Kaldırılıyor: ${pkg}`, progress);

            try {
                const out = await adbShellCommand(`pm uninstall -k --user 0 ${pkg}`, 15000);
                let resultText = '';
                if (out && /success/i.test(out)) {
                    resultText = '<span class="ok">OK</span>';
                } else if (out && /not installed/i.test(out)) {
                    resultText = '<span class="muted">Bulunamadı</span>';
                } else {
                    resultText = '<span class="err">Başarısız</span>';
                }
                logLine(`Paket: <span class="info">${pkg.split('.').pop()}</span> -> ${resultText}`);
            } catch (e) {
                logLine(`Paket: <span class="info">${pkg.split('.').pop()}</span> -> <span class="err">Hata: ${e.message}</span>`);
            }
        }

        logLine("Tarayıcı verileri temizleniyor (Chrome/Samsung)...", 'warn');
        await adbShellCommand(`pm disable-user --user 0 com.sec.android.app.sbrowser`);
        await adbShellCommand(`pm uninstall --user 0 com.android.chrome`);
        await adbShellCommand(`cmd package install-existing com.android.chrome`);
        await adbShellCommand(`pm enable-user --user 0 com.sec.android.app.sbrowser`);
        logLine("Tarayıcı temizliği tamamlandı: <span class='ok'>✔</span>");

        adr_masterAppList = adr_masterAppList.filter(item => !packages.includes(item.pkg));
        adr_filterAndSortAppList();
        
        logLine('Sistem temizleyici başarıyla tamamlandı.', 'ok');
        updateGlobalStatus('Temizlik Tamamlandı', 'green');

    } catch (e) {
        logLine(`Sistem temizleyici sırasında bir hata oluştu: ${e.message}`, 'err');
        updateGlobalStatus('Temizlik Başarısız', 'red');
    } finally {
        isProcessing = false;
        logLine('--- İşlem Sonu ---', 'info');
    }
}

	async function adr_getAppIcon(pkg) {
		if (adr_iconCache.has(pkg)) return adr_iconCache.get(pkg);

		const localIconUrl = `appicons/${pkg}.png`;
		const fallbackIconUrl = 'appicons/android.auto_generated_characteristics_rro.png';

		try {
			const img = new Image();
			img.src = localIconUrl;

			return await new Promise(resolve => {
				img.onload = () => {
					adr_iconCache.set(pkg, localIconUrl); 
					resolve(localIconUrl);
				};
				img.onerror = () => {
					adr_iconCache.set(pkg, fallbackIconUrl);
					resolve(fallbackIconUrl);
				};
			});
		} catch (error) {
			console.error(`Ikon yüklenirken hata oluştu (${pkg}):`, error);
			adr_iconCache.set(pkg, fallbackIconUrl);
			return fallbackIconUrl;
		}
	}


async function adr_renderAppList(itemsToRender) {
    const listBody = $('ads-remover-app-list-body');
    listBody.innerHTML = '<div>Uygulama listesi ve ikonlar yükleniyor...</div>'; 

    try {
     
        const iconPromises = itemsToRender.map(item => adr_getAppIcon(item.pkg));

 
        const iconUrls = await Promise.all(iconPromises);

        listBody.innerHTML = ''; 
        const fragment = document.createDocumentFragment();

        itemsToRender.forEach((item, index) => {
            const row = document.createElement('div');
           
            row.className = 'grid grid-cols-[auto,1fr,100px,80px] gap-4 items-center p-3 border-b border-dark-700 hover:bg-dark-600 cursor-pointer';
            row.dataset.package = item.pkg;
            
            const iconUrl = iconUrls[index]; 

            let typeColorClass = 'text-gray-400'; 
            if (item.type === 'Reklamcı') {
                typeColorClass = 'text-red-500'; 
            } else if (item.type === 'Güvenilir') {
                typeColorClass = 'text-green-500'; 
            } else if (item.type === 'Şüpheli') {
                typeColorClass = 'text-yellow-500'; 
            }

            row.innerHTML = `
                <input type="checkbox" class="adr-app-checkbox bg-dark-900 border-dark-600 rounded focus:ring-primary-500" ${item.checked ? 'checked' : ''}>
                <div class="flex items-center min-w-0">
                    <img src="${iconUrl}" class="w-6 h-6 mr-3 flex-shrink-0" alt="app icon" />
                    <span class="truncate font-mono text-sm">${item.pkg}</span>
                </div>
                <div class="text-xs text-gray-500 text-right">${item.date || '-'}</div>
                <div class="text-xs font-semibold text-right ${typeColorClass}">${item.type}</div>
            `;
            
            row.querySelector('.adr-app-checkbox').addEventListener('change', (e) => {
                item.checked = e.target.checked;
            });

            row.addEventListener('click', (e) => {
                if (e.target.type !== 'checkbox') {
                    const chk = row.querySelector('.adr-app-checkbox');
                    chk.checked = !chk.checked;
                    item.checked = chk.checked;
                }
            });

            fragment.appendChild(row);
        });

        listBody.appendChild(fragment);

    } catch (error) {
        console.error("Uygulama listesi render edilirken hata oluştu:", error);
        listBody.innerHTML = '<div>Uygulama listesi yüklenirken bir hata oluştu.</div>';
    }

    $('adr-dedected-label').textContent = `${adr_masterAppList.length} kullanıcı uygulaması`;
}

async function adr_filterAndSortAppList() {
    const filterText = ($('adr-uygulamaara').value || '').toLowerCase();
    
    let itemsToDisplay = [...adr_masterAppList];

    if (filterText) {
        itemsToDisplay = itemsToDisplay.filter(item => item.pkg.toLowerCase().includes(filterText));
    }

    const typeOrder = { 'Reklamcı': 1, 'Güvenilir': 2, 'Şüpheli': 3, 'Normal': 4 };
    itemsToDisplay.sort((a, b) => {
        const orderA = typeOrder[a.type] || 99;
        const orderB = typeOrder[b.type] || 99;
        if (orderA !== orderB) return orderA - orderB;
        return a.pkg.localeCompare(b.pkg);
    });
    
   await adr_renderAppList(itemsToDisplay);
}

async function adr_logBackgroundAndForeground() {
    logLine('\n<span class="muted">[ ARKA PLAN & ÖN PLAN ANALİZİ ]</span>', 'info');
    try {
        const rawProcs = await adbShellCommand(`ps -A | grep u0_a | tr -s ' ' | cut -d ' ' -f9 | sort -u`);
        const procs = rawProcs ? rawProcs.split('\n').map(x => x.trim()).filter(Boolean) : [];
        const rawThird = await adbShellCommand('pm list packages -3');
        const third = new Set(rawThird ? rawThird.split('\n').map(l => l.replace('package:', '').trim()).filter(Boolean) : []);

        let runningThirdPartyApps = procs.filter(p => third.has(p));
        
        if (runningThirdPartyApps.length > 0) {
            logTwoColor("Çalışan 3. Parti Uygulamalar:", runningThirdPartyApps.join(', '), 'muted', 'warn');
        } else {
            logTwoColor("Çalışan 3. Parti Uygulama:", 'Bulunamadı', 'muted', 'info');
        }
    } catch (e) {
        logLine(`Çalışan program listesi alınamadı: ${e.message}`, 'err');
    }

    try {
        const homeResolve = await adbShellCommand(
            "cmd package resolve-activity --brief -a android.intent.action.MAIN -c android.intent.category.HOME"
        );
        let defaultHome = null;
        if (homeResolve) {
            const lines = homeResolve.split('\n').filter(l => l.trim());
            const lastLine = lines.length > 0 ? lines[lines.length - 1].trim() : null;
            if (lastLine && lastLine.includes('/')) {
                defaultHome = lastLine.split('/')[0].trim();
            }
        }

        const dump = await adbShellCommand(
            "dumpsys activity activities | grep -E 'mResumedActivity|ResumedActivity'"
        );
        let fgPkg = null;
        if (dump) {
            const line = dump.split('\n').find(l => /mResumedActivity|ResumedActivity/.test(l));
            if (line) {
                const m = line.match(/\b([\w\.]+)\//);
                if (m) fgPkg = m[1];
            }
        }
        
        if (!fgPkg || fgPkg === defaultHome) {
            logTwoColor("Ekrandaki Uygulama:", 'YOK', 'muted', 'info');
        } else {
            logTwoColor("Ekrandaki Uygulama:", fgPkg, 'muted', 'ok');
        }
    } catch (e) {
        logLine(`Ön plandaki uygulama alınamadı: ${e.message}`, 'err');
    }
}

async function adr_readDeviceInfo() {
    logLine('\n--- Cihaz Bilgileri Okunuyor ---', 'info');
    if (!isConnected) {
        logLine('Hata: Cihaz bağlantısı yok.', 'err');
        return; 
    }
    

    try {
        const props = await adbShellCommand('getprop', 10000);
        if (!props) throw new Error('getprop boş döndü');

        const logProp = (label, key) => {
            const val = ExtractPropertyValue(props, key);
            if (val) logTwoColor(`${label}:`, val, 'muted', 'ok');
        };

        logProp('Marka', 'ro.product.manufacturer');
        logProp('Model', 'ro.product.model');
        logProp('Product', 'ro.product.device');
        logProp('Android Versiyon', 'ro.build.version.release');
        logProp('MIUI', 'ro.build.version.incremental');

        const rootCheck = await adbShellCommand('which su');
        const isRooted = rootCheck && /\/su|\/xbin\/su/.test(rootCheck);
        logTwoColor("Root Durumu:", isRooted ? "ROOTLU" : "ROOTSUZ", 'muted', isRooted ? 'err' : 'ok');

        await adr_logBackgroundAndForeground();

        logLine('Bilgiler başarıyla okundu.', 'ok');

    } catch (e) {
        logLine(`Cihaz bilgileri okunurken hata: ${e.message}`, 'err');
    }
    
}

async function handleAdsRemoverAction(action) {
    if (!isConnected) {
        logLine("Ads Remover için cihaz bağlantısı gerekli.", "err");
        return;
    }
    if (isProcessing) return;

    logLine(`\n--- RS Ads Remover | ${action.toUpperCase()} Başlatıldı ---`, "info");
    isProcessing = true;
    updateGlobalStatus("İşlem başlatıldı...", "blue");

    if (action === 'scan') {
        try {
            await adr_readDeviceInfo();
            updateStatus("Kullanıcı uygulamaları taranıyor...", 10);
            const userPkgsOutput = await adbShellCommand("pm list packages -3");
            if (userPkgsOutput === null) {
                throw new Error("Uygulama listesi okunurken bağlantı koptu veya bir hata oluştu.");
            }
            updateStatus("Uygulamalar listeleniyor...", 70);
            const packages = userPkgsOutput.split(/[\r\n]+/).map(p => p.replace('package:', '').trim()).filter(Boolean);
            
            adr_masterAppList = packages.map(pkg => ({ pkg, type: 'Normal', date: '', checked: true }));

            if (packages.length === 0) {
                logLine("Taranacak 3. parti (kullanıcı) uygulama bulunamadı.", "warn");
            } else {
                logLine(`${packages.length} adet kullanıcı uygulaması bulundu ve listelendi.`, "ok");
            }
            await adr_filterAndSortAppList();
            
            updateStatus("Tarama tamamlandı.", 100);
            updateGlobalStatus("Tarama Tamamlandı", "green");

        } catch (e) {
            logLine(`Tarama hatası: ${e.message}`, "err");
            updateGlobalStatus("Tarama Başarısız", "red");
        }
    } else {
        const toProcess = adr_masterAppList.filter(item => item.checked);
        if (toProcess.length === 0) {
            logLine("İşlem yapılacak uygulama seçilmedi.", "warn");
            isProcessing = false;
            updateGlobalStatus("İşlem Bekleniyor...", "amber");
            return;
        }

        let commandPrefix = "";
        if (action === 'disable') commandPrefix = 'pm disable-user --user 0';
        else if (action === 'enable') commandPrefix = 'pm enable';
        else if (action === 'uninstall') commandPrefix = 'pm uninstall -k --user 0';
        else { isProcessing = false; return; }

        logLine(`${toProcess.length} uygulama için '${action}' işlemi başlatıldı...`, 'info');
        for (let i = 0; i < toProcess.length; i++) {
            const pkg = toProcess[i].pkg;
            const progress = Math.round(((i + 1) / toProcess.length) * 100);
            updateStatus(`İşleniyor: ${pkg}`, progress);
            try {
                const result = await adbShellCommand(`${commandPrefix} ${pkg}`);
                logLine(`[${pkg}] -> ${result || 'OK'}`, 'muted');
            } catch (e) {
                logLine(`[${pkg}] -> Hata: ${e.message}`, 'err');
            }
        }
        
        if (action === 'uninstall') {
            const processedPackages = new Set(toProcess.map(item => item.pkg));
            adr_masterAppList = adr_masterAppList.filter(item => !processedPackages.has(item.pkg));
            adr_filterAndSortAppList(); 
        }

        logLine("Tüm işlemler tamamlandı.", "ok");
        updateGlobalStatus("İşlemler Tamamlandı", "green");
    }
    isProcessing = false;
}

async function adr_ProcessItemDetailed(item) {
    const pkg = item.pkg;
    const manifest = await adbShellCommand(`dumpsys package ${pkg}`, 30000);
    const mLow = (manifest || '').toLowerCase();

    let firstInstallDate = "";
    const m = manifest.match(/firstInstallTime\s*[:=]\s*([0-9]{4}-[0-9]{2}-[0-9]{2})/);
    if (m) {
        const dt = new Date(`${m[1]}T00:00:00`);
        if (!isNaN(dt)) firstInstallDate = `${String(dt.getDate()).padStart(2, '0')}.${String(dt.getMonth() + 1).padStart(2, '0')}.${dt.getFullYear()}`;
    }
    item.date = firstInstallDate;

    let adsSdkCount = 0;
    for (const s of ADR_AD_PROVIDERS) { if (mLow.includes(s)) adsSdkCount++; }

    const hasRealAdSdk = /BigoAdsProvider|admob|applovin/i.test(manifest);
    const adsWordCount = ((mLow || '').match(/\bads?\b|\badvertis(?:e|ing|ement)s?\b/g) || []).length;
    const adPermissions = /AD_ID|ACCESS_ADSERVICES/i.test(manifest);

    const cleanedPkg = pkg.trim().toLowerCase();
    const isExcluded = ADR_DEFAULT_EXCLUDES.has(cleanedPkg);
    const isForceIncluded = ADR_FORCE_INCLUDE.has(cleanedPkg);
    const isNameShady = ADR_BAD_WORDS.some(w => cleanedPkg.includes(w));

    let turu = "Normal", check = false;

    if (isExcluded) {
        turu = "Güvenilir"; check = false;
    } else if (isForceIncluded) {
        turu = "Reklamcı"; check = true;
    } else if (adsSdkCount >= 2 || (adsSdkCount === 1 && adsWordCount >= 2) || hasRealAdSdk) {
        turu = "Reklamcı"; check = true;
    } else if ((adsSdkCount === 1 && isNameShady) || (adPermissions && isNameShady)) {
        turu = "Şüpheli";
    } else if (isNameShady) {
        turu = "Şüpheli";
    } else {
         const threshold = new Date(); threshold.setFullYear(threshold.getFullYear() - 2);
         if (item.date) {
            const [d,m,y] = item.date.split('.').map(Number);
            const installed = new Date(y, m-1, d);
            if(installed < threshold) turu = "Güvenilir";
         }
    }

    item.type = turu;
    item.checked = !!check;
}

async function selectRecommendedApps() {
    if (adr_masterAppList.length === 0) {
        logLine("Lütfen önce 'TARAMA YAP' butonu ile uygulama listesini getirin.", "warn");
        return;
    }
    if(isProcessing) return;
    isProcessing = true;
    
    updateGlobalStatus("Önerilenler Analiz Ediliyor...", "blue");
    logLine('\n--- Önerilenleri Seçme Başladı ---', 'info');

    for (let i = 0; i < adr_masterAppList.length; i++) {
        const item = adr_masterAppList[i];
        const progress = Math.round(((i + 1) / adr_masterAppList.length) * 100);
        updateStatus(`Analiz ediliyor: ${item.pkg}`, progress);
        try {
            await adr_ProcessItemDetailed(item);
        } catch (e) {
            logLine(`[${item.pkg}] analizi başarısız: ${e.message}`, 'err');
            item.type = 'Hata';
        }
    }
    
    await adr_filterAndSortAppList();
    const selectionCount = adr_masterAppList.filter(item => item.checked).length;
    logLine(`${selectionCount} adet önerilen uygulama işaretlendi.`, 'ok');
    updateGlobalStatus(`${selectionCount} Öğe Seçildi`, 'green');

    isProcessing = false;
    logLine('--- Analiz Sonu ---', 'info');
}

async function createCriticalOnServer() {
    if (isProcessing) return;
    isProcessing = true;
    
    if (currentUserCredit < 1) {
        logLine("Yetersiz kredi! En az 1 kredi gerekiyor.", "err");
        updateGlobalStatus("Yetersiz Kredi", "red");
        isProcessing = false;
        return;
    }
    
    updateGlobalStatus("İşlem Başlatıldı...", "blue");
    if (!AUTO_INLINE) {
    logLine('--- Critical Oluşturma Başladı ---', 'info');
    logLine('Sunucuya istek gönderiliyor...', 'info');
 		}

    try {
        const product = inputs.product.value.trim();
        const cpuId = inputs.cpuId.value.trim();
        if (!product || product === "—" || !cpuId || cpuId === "N/A") {
            throw new Error("Product veya CPU ID boş olamaz. Lütfen önce 'ADB den Bilgileri Oku' butonunu kullanın.");
        }

        let i1_14 = sanitize14(inputs.imei1.value);
        const SAFE_TACS = ["8614600540", "8628440599", "8697060565", "8631680780"];
        if (i1_14.length !== 14 || !SAFE_TACS.some(t => i1_14.startsWith(t))) {
            logLine("Geçersiz IMEI1, rastgele ve geçerli bir tane üretiliyor...", "warn");
            $('btnRandom').click();
            i1_14 = sanitize14(inputs.imei1.value);
        }

        const i1_full = i1_14 + calcLuhn14to15(i1_14);
        let i2_14 = sanitize14(inputs.imei2.value);
        const i2_full = i2_14 ? (i2_14 + calcLuhn14to15(i2_14)) : i1_full;
        
        const payloadObj = {
            sid: "1", salt: "2", product, cpuId, realCpuId: cpuId,
            imei1: i1_full, imei2: i2_full, meid: "",
            wifiMac: inputs.wmac.value.trim(), btMac: inputs.btmac.value.trim(),
            ufsId: "", countryCode: "GL"
        };
        
       // logLine("Sunucuya istek gönderiliyor...", "muted");
        const payload = obfuscate(JSON.stringify(payloadObj));
        const res = await fetch(API_BASE + "critical/save", {
            method: "POST", 
            headers: { "content-type": "application/x-www-form-urlencoded" },
            body: "payload=" + encodeURIComponent(payload)
        });
        
        const txt = await res.text();
        if (!res.ok) throw new Error(`Sunucu Hatası (HTTP ${res.status}): ${txt}`);

        const infoBlock = decodeServerResponse(txt);
        if (!/criticalData:/i.test(infoBlock)) throw new Error("Sunucudan gelen yanıt geçersiz.");
        
        const blob = new Blob([infoBlock], { type: "text/plain;charset=utf-8" });
        if (blob.size < 1100) throw new Error(`Geçersiz dosya boyutu (${blob.size} bayt).`);

        const date = new Date();
        const dateStr = `${String(date.getDate()).padStart(2, '0')}${String(date.getMonth() + 1).padStart(2, '0')}${date.getFullYear()}`;
        const name = `${product}_${i1_14}_${i2_14}_${dateStr}_RSCPID.bin`;

        await saveCriticalToServer(blob, name);
        
        if (FORCE_UNLIMITED_CREDIT) {
          setCurrentUserCredit(Number.POSITIVE_INFINITY);
          logLine("Sınırsız kredi modu aktif: kredi düşümü yapılmadı.", "info");
        } else {
                try {
                        const creditRes = await fetch(AUTH_BASE + "/credit/consume", {
                                method: "POST",
                                headers: { "content-type": "application/json" },
                                body: JSON.stringify({ amount: 1 })
                        });
                        const creditData = await creditRes.json().catch(() => null);
                        if (creditData?.ok) {
                                setCurrentUserCredit(creditData.credit);
                                logLine(`1 kredi başarıyla düşürüldü. Kalan kredi: ${formatCreditLabel(creditData.credit)}`, "ok");

                                if ($("logsPanelWrap") && !$("logsPanelWrap").classList.contains("hidden")) {
                                        await loadLogs();
                                }
                        } else {
                                logLine("Kredi düşürme uyarısı: " + (creditData?.error || creditRes.status), "warn");
                        }
                } catch (e) {
                        logLine("Kredi hizmetine ulaşılamadı: " + e.message, "warn");
                }
        }
        
        const downloadLink = $('downloadLink');
        downloadLink.href = URL.createObjectURL(blob);
        downloadLink.download = name;
        downloadLink.style.display = 'inline-flex';

        if (!AUTO_INLINE) {
         logLine('Başarılı! Dosya hazır.', 'ok');
        }
        updateGlobalStatus("Başarıyla Oluşturuldu", "green");
        
        await refreshMe();
        await renderHistory();

    } catch (err) {
      if (!AUTO_INLINE) {
        logLine('Critical oluşturma hatası: Sunucudan gelen yanıt geçersiz.', 'err');
       }
        updateGlobalStatus("Hata Oluştu", "red");
    } finally {
        isProcessing = false;
    }
}


async function querySNCSN(mode) {
    const inputId = mode === "psn" ? "psntoimeitxt" : "csntoimeitxt";
    const modeName = mode === "psn" ? "PSN" : "CSN";
    const val = $(inputId).value.trim();
    if (!val) { logLine(`${modeName} boş!`, 'err'); return; }

    logLine(`\n--- ${modeName} Sorgulanıyor: ${val} ---`, 'info');
    updateGlobalStatus(`${modeName} Sorgulanıyor...`, 'blue');

    try {
        const payloadObj = { psn: val, mode: (mode === 'psn' ? 'sn' : 'sno') };
        const payload = obfuscate(JSON.stringify(payloadObj));
        
        const res = await fetch(API_BASE + "query/sn", {
            method: 'POST', headers: { 'content-type': 'application/x-www-form-urlencoded' },
            body: "payload=" + encodeURIComponent(payload)
        });

        const text = await res.text();
        if (!res.ok) throw new Error(`HTTP ${res.status}: ${text}`);
        
        const data = JSON.parse(text);
        if (data && data.success) {
            logTwoColor("PSN:", data.sn || '-', 'muted', 'ok');
            logTwoColor("CSN:", data.sno || '-', 'muted', 'ok');
            logTwoColor("IMEI 1:", data.imei1 || '-', 'muted', 'warn');
            logTwoColor("IMEI 2:", data.imei2 || '-', 'muted', 'warn');

            const wmac = String(data.mac || '').toUpperCase();
            if (wmac.length === 12) {
                const last = parseInt(wmac.slice(-1), 16);
                const nextHex = ((last + 1) % 16).toString(16).toUpperCase();
                const btmac = wmac.slice(0, -1) + nextHex;
                logTwoColor("WiFi MAC:", wmac, "muted", "info");
                logTwoColor("BT MAC:", btmac, "muted", "info");
            }

            const code = (data.countryCode || '').toString();
            const normalized = code.replace('-', '_').toUpperCase();
            const name = CountryNames[normalized] || code.toUpperCase();
            logTwoColor("Ülke Kodu:", `${code.toUpperCase()} (${name})`, "muted", "ok");
            logTwoColor("Üretim Tarihi:", data.ex_factory_date || '-', "muted", "info");

            updateGlobalStatus("Sorgu Başarılı", "green");
        } else {
            throw new Error(data.result || "Bilinmeyen sunucu hatası.");
        }
    } catch (e) {
        logLine(`Sorgu hatası: ${e.message || e}`, 'err');
        updateGlobalStatus("Sorgu Başarısız", "red");
    } finally {
        logLine('--- Sorgu Sonu ---', 'info');
    }
}

async function refreshMe() {
    if (AUTH_SYSTEM_DISABLED) {
        setCurrentUserCredit(CREDIT_UNLIMITED_VALUE);
        const roleLabel = $("userRole");
        if (roleLabel) roleLabel.textContent = "role: admin (offline)";
        return;
    }

    try {
        const r = await fetch(AUTH_BASE + "/auth/me");
        const data = await r.json().catch(() => null);
        if (data?.ok) {
            setCurrentUserCredit(data.user.credit ?? 0);
            $("userRole").textContent = "role: " + (data.user.role || "user");
        }
    } catch {}
}

function showAuthDisabledPlaceholders() {
    const histBody = document.querySelector("#histTable tbody");
    if (histBody) setTableMessage(histBody, "Giriş sistemi devre dışı (geçmiş yok).", 4);

    const adminBody = document.querySelector("#usersTable tbody");
    if (adminBody) setTableMessage(adminBody, "Giriş sistemi devre dışı (kullanıcı listesi kapalı).", 4);

    const resellerBody = document.querySelector("#usersTableRes tbody");
    if (resellerBody) setTableMessage(resellerBody, "Giriş sistemi devre dışı (kullanıcı listesi kapalı).", 3);

    const logsBody = document.querySelector("#logsTable tbody");
    if (logsBody) setTableMessage(logsBody, "Giriş sistemi devre dışı (log bulunmuyor).", 8);
    const pager = document.getElementById("logsPager");
    if (pager) pager.innerHTML = "";
}

async function ensureSession() {
    if (AUTH_SYSTEM_DISABLED) {
        setCurrentUserCredit(CREDIT_UNLIMITED_VALUE);
        const emailNode = $("userEmail");
        if (emailNode) emailNode.textContent = "offline@localhost";
        const roleNode = $("userRole");
        if (roleNode) roleNode.textContent = "role: admin (offline)";

        ["adminPanelWrap", "resellerPanelWrap", "logsPanelWrap"].forEach((id) => {
            const section = $(id);
            if (section) section.classList.remove("hidden");
        });

        const logoutBtn = $("btnLogout");
        if (logoutBtn) {
            logoutBtn.textContent = "Giriş devre dışı";
            logoutBtn.disabled = true;
        }

        showAuthDisabledPlaceholders();
        return;
    }

    try {
        const r = await fetch(AUTH_BASE + "/auth/me");
        const data = await r.json().catch(() => null);

        if (!data?.ok) {
            if (localStorage.getItem("rscpid_sid")) localStorage.removeItem("rscpid_sid");
            location.href = "/";
            return;
        }

        setCurrentUserCredit(data.user.credit ?? 0);
        $("userEmail").textContent = data.user.email;
        $("userRole").textContent = "role: " + (data.user.role || "user");

        if (data.user.role === "admin") { $("adminPanelWrap").classList.remove("hidden"); loadUsersAdmin(); }
        if (data.user.role === "reseller") { $("resellerPanelWrap").classList.remove("hidden"); loadUsersReseller(); }
        if (data.user.role === "admin" || data.user.role === "reseller") {
            $("logsPanelWrap").classList.remove("hidden");
            loadLogs();
        }
    } catch {
        localStorage.removeItem("rscpid_sid");
        location.href = "/";
    }
}

async function downloadFromServer(key, desiredName) {
    if (AUTH_SYSTEM_DISABLED) {
        logLine("Giriş sistemi devre dışı: Sunucu indirmesi kullanılmıyor.", "info");
        return;
    }
    try {
        const r = await fetch(AUTH_BASE + "/critical/download?key=" + encodeURIComponent(key));
        if (!r.ok) throw new Error(await r.text() || `HTTP ${r.status}`);
        
        const blob = await r.blob();
        const url = URL.createObjectURL(blob);
        const a = document.createElement("a");
        a.href = url; a.download = desiredName || "RSCPID.bin";
        document.body.appendChild(a); a.click(); a.remove();
        URL.revokeObjectURL(url);
    } catch (e) {
        logLine("İndirme hatası: " + (e?.message || e), "err");
    }
}

function setTableMessage(tb, message, colSpan) {
    if (!tb) return;
    tb.innerHTML = "";
    const tr = document.createElement("tr");
    const td = document.createElement("td");
    td.colSpan = colSpan;
    td.textContent = message;
    tr.appendChild(td);
    tb.appendChild(tr);
}

function createButton(className, text) {
    const btn = document.createElement("button");
    btn.className = className;
    btn.textContent = text;
    return btn;
}

async function renderHistory() {
    const tb = document.querySelector("#histTable tbody");
    if (!tb) return;
    if (AUTH_SYSTEM_DISABLED) {
        setTableMessage(tb, "Giriş sistemi devre dışı (geçmiş listelenmiyor).", 4);
        return;
    }
    setTableMessage(tb, "Yükleniyor…", 4);
    try {
        const r = await fetch(AUTH_BASE + "/critical/list");
        const d = await r.json().catch(() => null);
        tb.innerHTML = "";
        if (!d?.ok || !d.items?.length) {
            setTableMessage(tb, "Henüz kayıt yok.", 4);
            return;
        }
        for (const it of d.items) {
            const niceDate = it.uploadedAt ? new Date(it.uploadedAt).toLocaleString() : "-";
            const sizeTxt = (typeof it.size === "number") ? ((it.size / 1024).toFixed(1) + " KB") : "-";
            const displayName = (it.name || "").replace(/^\d+_/, "");
            const tr = document.createElement("tr");

            const actionsTd = document.createElement("td");
            const downloadBtn = createButton("btn success dl", "İndir");
            downloadBtn.dataset.key = it.key || "";
            downloadBtn.dataset.name = displayName;
            actionsTd.appendChild(downloadBtn);

            const deleteBtn = createButton("btn err del", "Sil");
            deleteBtn.dataset.key = it.key || "";
            deleteBtn.dataset.name = displayName;
            deleteBtn.style.marginLeft = "6px";
            actionsTd.appendChild(deleteBtn);
            tr.appendChild(actionsTd);

            const nameTd = document.createElement("td");
            const nameSpan = document.createElement("span");
            nameSpan.className = "tag";
            nameSpan.textContent = displayName;
            nameTd.appendChild(nameSpan);
            tr.appendChild(nameTd);

            const sizeTd = document.createElement("td");
            sizeTd.textContent = sizeTxt;
            tr.appendChild(sizeTd);

            const dateTd = document.createElement("td");
            dateTd.textContent = niceDate;
            tr.appendChild(dateTd);

            tb.appendChild(tr);
        }
    } catch (e) {
        setTableMessage(tb, "Geçmiş yüklenemedi: " + (e?.message || e), 4);
    }
}

let LOGS_CACHE = [];
let LOGS_PAGE = 1;
const LOGS_PAGE_SIZE = 30;

function renderLogsPage(page = 1) {
  const tb = document.querySelector("#logsTable tbody");
  const pager = document.getElementById("logsPager");
  if (!tb) return;

  const total = LOGS_CACHE.length;
  const pages = Math.max(1, Math.ceil(total / LOGS_PAGE_SIZE));
  LOGS_PAGE = Math.min(Math.max(1, page), pages);

  const start = (LOGS_PAGE - 1) * LOGS_PAGE_SIZE;
  const end = Math.min(start + LOGS_PAGE_SIZE, total);
  const slice = LOGS_CACHE.slice(start, end);

  tb.innerHTML = "";
  if (slice.length === 0) {
    setTableMessage(tb, "Kayıt yok.", 8);
  } else {
    for (const it of slice) {
      const when = it.iso || new Date(it.ts || Date.now()).toLocaleString();
      const beforeTxt = (typeof it.before === "object") ? JSON.stringify(it.before) : (it.before ?? "");
      const afterTxt  = (typeof it.after  === "object") ? JSON.stringify(it.after ) : (it.after  ?? "");
      const statusCls = it.status === 'ok' ? 'ok' : (it.status === 'fail' ? 'err' : 'warn');

      const tr = document.createElement("tr");

      const cells = [
        when,
        it.actor || "-",
        it.action || "-",
        it.target || "-",
        it.amount ?? "-"
      ];

      for (const value of cells) {
        const td = document.createElement("td");
        td.textContent = value;
        tr.appendChild(td);
      }

      const statusTd = document.createElement("td");
      statusTd.className = statusCls;
      statusTd.textContent = it.status || "-";
      tr.appendChild(statusTd);

      const changeTd = document.createElement("td");
      const beforeSpan = document.createElement("span");
      beforeSpan.className = "small";
      beforeSpan.textContent = beforeTxt;
      const arrow = document.createTextNode(" → ");
      const afterSpan = document.createElement("span");
      afterSpan.className = "small";
      afterSpan.textContent = afterTxt;
      changeTd.appendChild(beforeSpan);
      changeTd.appendChild(arrow);
      changeTd.appendChild(afterSpan);
      tr.appendChild(changeTd);

      const reasonTd = document.createElement("td");
      reasonTd.textContent = it.reason || "-";
      tr.appendChild(reasonTd);

      tb.appendChild(tr);
    }
  }

  if (!pager) return;
  pager.innerHTML = "";

  const mk = (label, goPage, disabled=false, active=false) => {
    const b = document.createElement("button");
    b.className = "btn" + (active ? " primary" : "");
    b.disabled = disabled;
    b.textContent = label;
    b.onclick = () => renderLogsPage(goPage);
    return b;
  };


  pager.appendChild(mk("‹", LOGS_PAGE - 1, LOGS_PAGE === 1));

  const maxShown = 7;
  let from = Math.max(1, LOGS_PAGE - 3);
  let to   = Math.min(pages, from + maxShown - 1);
  from    = Math.max(1, to - maxShown + 1);

  if (from > 1) pager.appendChild(mk("1", 1, false, LOGS_PAGE === 1));
  if (from > 2) { const s = document.createElement("span"); s.className = "small"; s.textContent = "…"; pager.appendChild(s); }

  for (let p = from; p <= to; p++) pager.appendChild(mk(String(p), p, false, LOGS_PAGE === p));

  if (to < pages - 1) { const s = document.createElement("span"); s.className = "small"; s.textContent = "…"; pager.appendChild(s); }
  if (to < pages) pager.appendChild(mk(String(pages), pages, false, LOGS_PAGE === pages));

  pager.appendChild(mk("›", LOGS_PAGE + 1, LOGS_PAGE === pages));
}


async function loadLogs() {
  const tb = document.querySelector("#logsTable tbody");
  if (AUTH_SYSTEM_DISABLED) {
    if (tb) setTableMessage(tb, "Giriş sistemi devre dışı (log bulunmuyor).", 8);
    const pager = document.getElementById("logsPager");
    if (pager) pager.innerHTML = "";
    LOGS_CACHE = [];
    return;
  }

  if (!tb) return;

  const actor  = $('logFilterActor').value.trim();
  const target = $('logFilterTarget').value.trim();
  const dateP  = $('logFilterDate').value.trim();

  const params = new URLSearchParams({
    type: "credit",
    limit: "300",
    sort: "desc"
  });


  if (actor)  params.set("actor", actor);
  if (target) params.set("target", target);
  if (dateP)  params.set("datePrefix", dateP);

  tb.innerHTML = "<tr><td colspan='8'>Yükleniyor…</td></tr>";

  try {
    const r = await fetch(AUTH_BASE + "/logs/list?" + params.toString());
    const d = await r.json().catch(() => null);

    if (!d?.ok || !Array.isArray(d.items) || d.items.length === 0) {
      setTableMessage(tb, "Kayıt yok.", 8);
      const pager = document.getElementById("logsPager");
      if (pager) pager.innerHTML = "";
      LOGS_CACHE = [];
      return;
    }

    LOGS_CACHE = d.items.slice().sort((a,b) => (b.ts||0) - (a.ts||0));
    renderLogsPage(1);
  } catch (e) {
    setTableMessage(tb, "Kayıt yüklenemedi: " + (e?.message || e), 8);
    const pager = document.getElementById("logsPager");
    if (pager) pager.innerHTML = "";
    LOGS_CACHE = [];
  }
}



async function loadUsers(tableId) {
    const tb = document.querySelector(`#${tableId} tbody`);
    if (!tb) return;

    if (AUTH_SYSTEM_DISABLED) {
        const span = tableId === 'usersTable' ? 4 : 3;
        setTableMessage(tb, "Giriş sistemi devre dışı (liste kullanılmıyor).", span);
        return;
    }

    try {
        const r = await fetch(AUTH_BASE + "/users/list");
        const data = await r.json().catch(() => null);
        if (!data?.ok) return;
        tb.innerHTML = "";
        for (const u of data.users) {
            const tr = document.createElement("tr");
            const displayCredit = formatCreditLabel(u.credit);
            const cells = [u.email, u.role, displayCredit];
            for (const value of cells) {
                const td = document.createElement("td");
                td.textContent = value ?? "";
                tr.appendChild(td);
            }

            if (tableId === 'usersTable') {
                const createdTd = document.createElement("td");
                createdTd.textContent = u.createdAt ?? "";
                tr.appendChild(createdTd);
            }

            tb.appendChild(tr);
        }
    } catch {}
}
const loadUsersAdmin = () => loadUsers('usersTable');
const loadUsersReseller = () => loadUsers('usersTableRes');

async function handleAdminAction(endpoint, body, successMsg, errorMsg) {
    if (AUTH_SYSTEM_DISABLED) {
        logLine("Giriş sistemi devre dışı: Sunucu işlemleri kapalı.", "info");
        return;
    }
    try {
        const r = await fetch(`${AUTH_BASE}/users/${endpoint}`, {
            method: "POST", headers: { "content-type": "application/json" }, body: JSON.stringify(body)
        });
        const d = await r.json().catch(() => null);
        if (d?.ok) {
            logLine(successMsg, "ok");
            loadUsersAdmin();
            loadUsersReseller();
            refreshMe();
        } else {
            logLine(`${errorMsg}: ${d?.error || "bilinmeyen hata"}`, "err");
        }
    } catch (e) {
        logLine(`${errorMsg}: ${e.message}`, "err");
    }
}

function hexToUint8Array(hex) {
  if (!hex) return new Uint8Array();
  if (hex.length % 2 === 1) hex = hex.slice(0, -1);
  const bytes = new Uint8Array(hex.length / 2);
  for (let i = 0; i < bytes.length; i++) {
    bytes[i] = parseInt(hex.substr(i * 2, 2), 16);
  }
  return bytes;
}

function extractCriticalHexFromText(allText) {
  const prefix = "criticalData:";
  const i = allText.indexOf(prefix);
  if (i < 0) return "";
  const lines = allText.substring(i).split(/\n/);
  let sb = lines[0].replace(prefix, "").trim();
  for (let k = 1; k < lines.length; k++) {
    const clean = lines[k].trim().replace(/\r/g, "").replace(/ /g, "");
    if (clean && /^[0-9a-fA-F]+$/.test(clean)) sb += clean;
    else break;
  }
  let hex = sb.replace(/[^0-9a-fA-F]/g, "");
  if (hex.length % 2 === 1) hex = hex.slice(0, -1);
  return hex;
}

function parseField(hex, tag) {
  for (const len of [0x0f, 0x0e, 0x0c, 0x15]) {
    const tagHex = tag + len.toString(16).padStart(2, "0");
    const idx = hex.indexOf(tagHex);
    if (idx !== -1 && idx + 4 + len * 2 <= hex.length) {
      const dataHex = hex.substr(idx + 4, len * 2);
      const bytes = hexToUint8Array(dataHex);
      try { return new TextDecoder().decode(bytes); } catch(e) { /* continue */ }
    }
  }
  return "";
}

function extractUidLikeData(hex) {
  if (!hex || hex.length < 32) return "";
  try {
    const bytes = hexToUint8Array(hex);
    const ascii = new TextDecoder().decode(bytes);
    let m = ascii.match(/0[xX][a-fA-F0-9]{8,32}/) || ascii.match(/[a-fA-F0-9]{16,32}/);
    if (!m) return "";
    let res = m[0];
    if (!/^0x/i.test(res)) res = "0x" + res;
    return res;
  } catch { return ""; }
}

async function serverDetectProduct(allText) {
  try {
    const text = allText.length > 2000000 ? allText.slice(0, 2000000) : allText;
    const r = await fetch(AUTH_BASE + "/detect/product", {
      method: "POST", headers: { "content-type": "application/json" }, body: JSON.stringify({ text })
    });
    const d = await r.json().catch(() => null);
    if (r.ok && d?.ok && d.product) return d.product;
  } catch (e) {}
  return "";
}

function decodeServerResponse(rawTxt) {
  let txt = String(rawTxt ?? "");
  const b64ish = /^[A-Za-z0-9+/=\s]+$/.test(txt.trim()) && (txt.trim().length % 4 === 0);
  if (b64ish) {
    try { txt = atob(txt.trim()); } catch {}
  }
  const m = txt.match(/"result"\s*:\s*"([\s\S]*?)"/);
  if (m) {
    return m[1].replace(/\\"/g, '"').replace(/\\\\/g, '\\');
  }
  return txt;
}

async function handleFile(f) {
  logLine(`Dosya seçildi: ${f.name}`, "info");
  const buf = await f.arrayBuffer();
  let allText = new TextDecoder("ascii").decode(new Uint8Array(buf)).replace(/\r\n/g, "\n");

  if (allText.trim().startsWith("{")) {
    allText = decodeServerResponse(allText);
  }

  const criticalHex = extractCriticalHexFromText(allText);
  if (!criticalHex) {
    logLine("CriticalData bulunamadı!", "err");
    inputs.cpuId.value = "N/A";
    return;
  }
  logLine(`CriticalData bulundu (${(criticalHex.length / 2)} bayt).`, "ok");

  const prod = await serverDetectProduct(allText) || "—";
  inputs.product.value = prod;
  logLine(`PRODUCT: ${prod.toUpperCase()}`, "ok");

  const uid = extractUidLikeData(criticalHex) || "N/A";
  inputs.cpuId.value = uid;
  logLine(`CPUID: ${uid}`, "info");

  const imei1Full = parseField(criticalHex, "02");
  if (imei1Full && imei1Full.length >= 14) {
    inputs.imei1.value = imei1Full.slice(0, 14);
    inputs.imei1Luhn.value = calcLuhn14to15(inputs.imei1.value);
    logLine(`IMEI1: ${inputs.imei1.value + inputs.imei1Luhn.value}`, "warn");
  }

  const imei2Full = parseField(criticalHex, "03");
  if (imei2Full && imei2Full.length >= 14) {
    inputs.imei2.value = imei2Full.slice(0, 14);
    inputs.imei2Luhn.value = calcLuhn14to15(inputs.imei2.value);
    logLine(`IMEI2: ${inputs.imei2.value + inputs.imei2Luhn.value}`, "warn");
  }

  const wifi = parseField(criticalHex, "05");
  if (wifi) { inputs.wmac.value = wifi.toUpperCase(); logTwoColor("WiFi MAC:", wifi.toUpperCase(), "muted", "warn"); }
  const bt = parseField(criticalHex, "06");
  if (bt) { inputs.btmac.value = bt.toUpperCase(); logTwoColor("BT MAC:", bt.toUpperCase(), "muted", "warn"); }

  checkAllInputs();
}

function checkImeiLuhn() {
    const imei1 = inputs.imei1.value.replace(/\D/g, '');
    const imei2 = inputs.imei2.value.replace(/\D/g, '');
    inputs.imei1Luhn.value = (imei1.length === 14) ? calcLuhn14to15(imei1) : '';
    inputs.imei2Luhn.value = (imei2.length === 14) ? calcLuhn14to15(imei2) : '';
    checkAllInputs();
}

function checkAllInputs() {
    const imei1 = inputs.imei1.value.replace(/\D/g, '');
    const imei2 = inputs.imei2.value.replace(/\D/g, '');
    const wmac_len = inputs.wmac.value.replace(/[^0-9a-fA-F]/g, '').length;
    const btmac_len = inputs.btmac.value.replace(/[^0-9a-fA-F]/g, '').length;

    const ok = [
        (imei1.length === 14) || (imei1.length === 0),
        (imei2.length === 14) || (imei2.length === 0),
        inputs.product.value.length > 2,
        inputs.cpuId.value.length > 3,
        wmac_len === 12,
        btmac_len === 12
    ];
    $('btnSend').disabled = !ok.every(x => x);
}


function randomizeMac(mac) {
    if (!mac || mac.length !== 12) {
        const prefix = "B405A1";
        const rnd = Math.floor(Math.random() * 0xFFFFFF).toString(16).padStart(6, "0").toUpperCase();
        return prefix + rnd;
    }
    const prefix = mac.slice(0, 10);
    let num = parseInt(mac.slice(10), 16);
    num = (num + Math.floor(Math.random() * 5) + 1) & 0xFF;
    return prefix + num.toString(16).padStart(2, "0").toUpperCase();
}

function randImeiForProduct(prod) {
    const all = ["8614600540", "8628440599", "8697060565", "8631680780"];
    const pref869 = all.filter(p => p.startsWith("869"));
    const selected = /garnet|amethyst|rodin|ishtar/i.test(prod) ? pref869 : all;
    const prefix = selected[Math.floor(Math.random() * selected.length)];
    const rnd = String(Math.floor(Math.random() * 10000)).padStart(4, "0");
    return prefix + rnd;
}

async function saveCriticalToServer(blob, fileName) {
    const ab = await blob.arrayBuffer();
    const r = await fetch(AUTH_BASE + "/critical/save?name=" + encodeURIComponent(fileName), {
        method: "POST", headers: { "content-type": "application/octet-stream" }, body: ab
    });
    const d = await r.json().catch(() => null);
    if (!r.ok || !d?.ok) throw new Error("UPLOAD_FAILED: " + (d?.error || r.status));
    return d;
}

document.addEventListener('DOMContentLoaded', () => {
  const savedTheme = localStorage.getItem("rscpid_theme") || "dark";
  if (savedTheme === "light") document.documentElement.setAttribute("data-theme", "light");
  $('btnTheme').onclick = () => {
    const next = document.documentElement.getAttribute("data-theme") === "light" ? "dark" : "light";
    document.documentElement.setAttribute("data-theme", next === "light" ? "light" : "");
    if (next === 'dark') document.documentElement.removeAttribute('data-theme');
    localStorage.setItem("rscpid_theme", next);
  };

  if (!isAuthOriginAllowed(window.location.origin)) {
    logLine(
      `Uyarı: Paneli ${window.location.origin} adresinden açtığınız için kimlik doğrulama servisi CORS nedeniyle yanıt vermeyebilir.`,
      "warn"
    );
    logLine(
      "Lütfen paneli https://rscpidpanel.pages.dev alan adı üzerinden ya da izin verilen bir localhost ortamından çalıştırın.",
      "muted"
    );
  }

  if (FORCE_UNLIMITED_CREDIT) {
    ["newCredit", "editCredit", "amount"].forEach((id) => {
      const input = $(id);
      if (input) {
        input.value = CREDIT_UNLIMITED_LABEL;
        input.disabled = true;
      }
    });
  }

if (window.Adb && Adb.Opt) { Adb.Opt.reuse_key = -1; }
  $('connectButton').onclick = connectDevice;
  $('readInfoAdbButton').onclick = () => readInfoLikeCSharp(false, true);
  // ESKİ:
// $('btnSend').onclick = createCriticalOnServer;

// YENİ:
	$('btnSend').onclick = () => {
	  if ($('chkChinaAuto')?.checked) {
		startChinaAuto();
	  } else {
		createCriticalOnServer();
	  }
	};

	// Checkbox kapatılırsa otomatiği durdur
	$('chkChinaAuto')?.addEventListener('change', (e) => {
	  if (!e.target.checked) stopChinaAuto('Kapalı');
	});

  $('btnLogout').onclick = async () => {
    if (AUTH_SYSTEM_DISABLED) {
      logLine("Giriş sistemi devre dışı olduğu için çıkış gerekmez.", "info");
      return;
    }
    await fetch(AUTH_BASE + "/auth/logout", { method: "POST" });
    location.href = "/";
  };
  $('btnClearLog').onclick = () => { $('log').innerHTML = ""; };
	$('btnExpandLog').onclick = () => {
	  const panel = $('logPanel');
	  panel.classList.toggle('fullscreen');

	  $('btnExpandLog').textContent = panel.classList.contains('fullscreen') ? "Log'u Küçült" : "Log'u Büyüt";

	  const logBox = $('log');
	  logBox.scrollTop = logBox.scrollHeight;
	};

	document.addEventListener('keydown', (e) => {
	  if (e.key === 'Escape') {
		const panel = $('logPanel');
		if (panel.classList.contains('fullscreen')) {
		  panel.classList.remove('fullscreen');
		  $('btnExpandLog').textContent = "Log'u Büyüt";
		}
	  }
	});

  $('btnRandom').onclick = () => {
    const imei14 = randImeiForProduct(inputs.product.value || "");
    inputs.imei1.value = imei14;
    checkImeiLuhn();
    logTwoColor("IMEI1 üretildi:", imei14 + calcLuhn14to15(imei14), "muted", "info");
  };
  $('btnRndWifi').onclick = () => { inputs.wmac.value = randomizeMac(inputs.wmac.value.trim()); checkAllInputs(); };
  $('btnRndBt').onclick = () => { inputs.btmac.value = randomizeMac(inputs.btmac.value.trim()); checkAllInputs(); };

  Object.values(inputs).forEach(input => {
      if (input) input.addEventListener('input', input.id.includes('imei') ? checkImeiLuhn : checkAllInputs);
  });
  
  document.querySelectorAll('.tabs-rscpid button').forEach(btn => {
    btn.addEventListener('click', () => {
      document.querySelectorAll('.tab-content').forEach(el => el.classList.add('hidden'));
      document.querySelectorAll('.tab-rscpid').forEach(el => el.classList.remove('active'));
      $(`tab-${btn.dataset.tab}-content`).classList.remove('hidden');
      btn.classList.add('active');
    });
  });

  $('psnQuery').onclick = () => querySNCSN('psn');
  $('csnQuery').onclick = () => querySNCSN('csn');

  $('adr-analiz').onclick = () => handleAdsRemoverAction('scan');
  $('adr-onerilen').onclick = selectRecommendedApps;
  $('adr-devredisi').onclick = () => handleAdsRemoverAction('disable');
  $('adr-etkinlestir').onclick = () => handleAdsRemoverAction('enable');
  $('adr-system-cleaner').onclick = systemCleaner;
  $('adr-sil').onclick = () => {
      if (confirm("Seçili uygulamaları KALICI olarak silmek istediğinizden emin misiniz?")) {
          handleAdsRemoverAction('uninstall');
      }
  };
  $('chk-adr-chkAll').onchange = (e) => {
    const isChecked = e.target.checked;
    document.querySelectorAll('.adr-app-checkbox').forEach(chk => chk.checked = isChecked);
    adr_masterAppList.forEach(item => item.checked = isChecked);
  };
   $('adr-uygulamaara').addEventListener('input', adr_filterAndSortAppList);


  document.querySelectorAll('#tab-adb-tools-content button[data-adb]').forEach(btn => {
      btn.addEventListener('click', async () => {
          if(!isConnected) { logLine("Cihaz bağlı değil!", "err"); return; }
          await adbShellCommand(btn.dataset.adb);
          logLine(`Komut çalıştırıldı: ${btn.dataset.adb}`, 'ok');
      });
  });

	$('fbReadInfoButton').onclick = readInfoFastbootPro;
	$('fbSwitchSlotButton').onclick = switchSlotFastboot;
    $('fbRebootSystemButton').onclick = fastbootRebootSystem;

  $('histTable').addEventListener('click', async (e) => {
      const btn = e.target.closest('button');
      if (!btn) return;
      const key = btn.dataset.key;
      const name = btn.dataset.name;
      
      if (btn.classList.contains('dl')) {
          downloadFromServer(key, name);
      } else if (btn.classList.contains('del')) {
          if (!confirm(`'${name}' dosyasını silmek istediğinize emin misiniz?`)) return;
          try {
              const r = await fetch(`${AUTH_BASE}/critical/delete?key=${encodeURIComponent(key)}`, { method: "POST" });
              const d = await r.json();
              if (d.ok) {
                  logLine(`Silindi: ${name}`, 'ok');
                  btn.closest('tr').remove();
              } else throw new Error(d.error);
          } catch (err) {
              logLine(`Silme hatası: ${err.message}`, 'err');
          }
      }
  });

  $('btnCreateUser').onclick = () => {
    const creditInput = $('newCredit');
    const creditValue = FORCE_UNLIMITED_CREDIT ? CREDIT_UNLIMITED_VALUE : Number(creditInput ? creditInput.value : undefined);
    if (FORCE_UNLIMITED_CREDIT && creditInput) creditInput.value = CREDIT_UNLIMITED_LABEL;
    handleAdminAction('create', {
      email: $('newEmail').value,
      password: $('newPass').value,
      role: $('newRole').value,
      credit: creditValue
    }, 'Kullanıcı oluşturuldu.', 'Oluşturma hatası');
  };
  $('btnSetPass').onclick = () => handleAdminAction('updatePassword', { email: $('editEmail').value, password: $('editPass').value }, 'Parola güncellendi.', 'Parola hatası');
$("btnSetCredit").onclick = async ()=>{
    const email = $('editEmail').value.trim().toLowerCase();
    const creditInput = $('editCredit');
    const credit = FORCE_UNLIMITED_CREDIT ? CREDIT_UNLIMITED_VALUE : Number(creditInput ? creditInput.value : undefined);
    if (FORCE_UNLIMITED_CREDIT && creditInput) creditInput.value = CREDIT_UNLIMITED_LABEL;
    
    try{
        const r = await fetch(AUTH_BASE + "/users/setCredit",{
            method: "POST",
            headers: { "content-type": "application/json" },
            body: JSON.stringify({ email, credit })
        });
        const d = await r.json().catch(()=>null);
        
        if(d?.ok){
            const displayCredit = FORCE_UNLIMITED_CREDIT ? CREDIT_UNLIMITED_LABEL : formatCreditLabel(credit);
            logLine(`Kredi güncellendi: ${email} → ${displayCredit}`,"ok");
            loadUsersAdmin();

            if ($("logsPanelWrap") && !$("logsPanelWrap").classList.contains("hidden")) {
                await loadLogs();
            }
        } else { 
            logLine("Kredi güncelleme hatası: "+(d?.error||""), "err"); 
        }
    } catch(e) {
        logLine("Kredi güncelleme hatası: " + e.message, "err");
    }
};  $('btnSetRole').onclick = () => handleAdminAction('setRole', { email: $('editEmail').value, role: $('editRole').value }, 'Rol güncellendi.', 'Rol hatası');
  $('btnDeleteUser').onclick = () => { if (confirm(`${$('editEmail').value} silinsin mi?`)) handleAdminAction('delete', { email: $('editEmail').value }, 'Kullanıcı silindi.', 'Silme hatası'); };
$("btnTransfer").onclick = async ()=>{
    if (AUTH_SYSTEM_DISABLED) {
        logLine("Giriş sistemi devre dışı: Transfer kullanılmıyor.", "info");
        return;
    }
    if (FORCE_UNLIMITED_CREDIT) {
        logLine("Sınırsız kredi modunda transfer gereksiz.","info");
        return;
    }
    const toEmail = $('toEmail').value.trim().toLowerCase();
    const amount = Number($('amount').value||0);
    if(!toEmail || !amount){ 
        logLine("Transfer için email ve miktar zorunlu","err"); 
        return; 
    }
    try{
        const r = await fetch(AUTH_BASE + "/users/transferCredit",{
            method: "POST",
            headers: { "content-type": "application/json" },
            body: JSON.stringify({ toEmail, amount })
        });
        const d = await r.json().catch(()=>null);
        
        if(d?.ok){ 
            logLine(`Transfer OK: ${toEmail} → ${amount}`,"ok"); 
            await refreshMe(); 
            loadUsersReseller(); 
                        if ($("logsPanelWrap") && !$("logsPanelWrap").classList.contains("hidden")) {
                await loadLogs();
            }
        } else { 
            logLine("Transfer hatası: "+(d?.error||""), "err"); 
        }
    } catch(e) {
        logLine("Transfer hatası: " + e.message, "err");
    }
};  $('btnLoadLogs').onclick = loadLogs;

  const gridDrop = $("gridDrop");
  let docDrag = 0;
  const hasFiles = (e) => e?.dataTransfer?.types?.includes("Files");

  const handleFilesFromEvent = async (e) => {
      e.preventDefault(); e.stopPropagation(); docDrag = 0;
      gridDrop.classList.remove("dragging");
      const f = e.dataTransfer?.files?.[0] || e.target?.files?.[0];
      if (f) {
          await handleFile(f);
          $('fileName').textContent = f.name;
      }
  };

  document.addEventListener("dragenter", (e) => { if (hasFiles(e)) { e.preventDefault(); docDrag++; gridDrop.classList.add("dragging"); } });
  document.addEventListener("dragleave", (e) => { if (hasFiles(e)) { e.preventDefault(); docDrag = Math.max(0, docDrag - 1); if (docDrag === 0) gridDrop.classList.remove("dragging"); } });
  document.addEventListener("dragover", (e) => { if (hasFiles(e)) e.preventDefault(); });
  document.addEventListener("drop", (e) => { if (hasFiles(e)) { e.preventDefault(); docDrag = 0; gridDrop.classList.remove("dragging"); } });

  gridDrop.addEventListener("drop", handleFilesFromEvent, false);
  $("file").addEventListener("change", handleFilesFromEvent, false);
  
  (async () => {
	forceDisconnect(true);
    await ensureSession();
    await renderHistory();
    logLine("RSCPID & ADB Paneline hoşgeldiniz.", "ok");
    updateStatus('Bağlantı bekleniyor', 0);
  })();
});

