
(function(root, factory) {
    if (typeof define === 'function' && define.amd) {
        define([], factory);
    } else if (typeof exports === 'object') {
        module.exports = factory();
    } else {
        root.Adb = factory();
    }
}(this, function() {
    'use strict';
    let Adb = {};
    let Fastboot = {};
    Adb.Opt = {};
    Adb.Opt.debug = false;
    Adb.Opt.dump = false;
    Adb.Opt.key_size = 2048;
    Adb.Opt.reuse_key = -1;
    Adb.Opt.use_checksum = true;
    let db = init_db();
    let keys = db.then(load_keys);
    Adb.open = function(transport) {
        if (transport == "WebUSB")
            return Adb.WebUSB.Transport.open();
        throw new Error("Unsupported transport: " + transport);
    };
    Adb.WebUSB = {};
    Adb.WebUSB.Transport = function(device) {
        this.device = device;
        if (Adb.Opt.debug)
            console.log(this);
    };
	Adb.WebUSB.Transport.open = function () {
	  const filters = [
		{ classCode: 255, subclassCode: 66, protocolCode: 1 },
		{ classCode: 255, subclassCode: 66, protocolCode: 3 },
		{ vendorId: 0x18D1 },
		{ vendorId: 0x0E8D },
		{ vendorId: 0x0525 },
	  ];


	  return navigator.usb.getDevices().then(async (devices) => {
		let device = null;

		if (devices && devices.length === 1) {
		  device = devices[0];
		} else if (devices && devices.length > 1) {
		  device = await navigator.usb.requestDevice({ filters });
		} else {
		  device = await navigator.usb.requestDevice({ filters });
		}

		await device.open();
		return new Adb.WebUSB.Transport(device);
	  });
	};

    Adb.WebUSB.Transport.prototype.close = function() {
        return this.device.close();
    };
    Adb.WebUSB.Transport.prototype.reset = function() {
        return this.device.reset();
    };
    Adb.WebUSB.Transport.prototype.send = function(ep, data) {
        if (Adb.Opt.dump)
            hexdump(new DataView(data), "" + ep + "==> ");
        return this.device.transferOut(ep, data);
    };
    Adb.WebUSB.Transport.prototype.receive = function(ep, len) {
        return this.device.transferIn(ep, len)
            .then(response => {
                if (Adb.Opt.dump)
                    hexdump(response.data, "<==" + ep + " ");
                return response.data;
            });
    };
    Adb.WebUSB.Transport.prototype.find = function(filter) {
        for (let i in this.device.configurations) {
            let conf = this.device.configurations[i];
            for (let j in conf.interfaces) {
                let intf = conf.interfaces[j];
                for (let k in intf.alternates) {
                    let alt = intf.alternates[k];
                    if (filter.classCode == alt.interfaceClass &&
                        filter.subclassCode == alt.interfaceSubclass &&
                        filter.protocolCode == alt.interfaceProtocol) {
                        return { conf: conf, intf: intf, alt: alt };
                    }
                }
            }
        }
        return null;
    }
    Adb.WebUSB.Transport.prototype.getDevice = function(filter) {
        let match = this.find(filter);
        if (match === null) {
           return Promise.reject(new Error("Cihazın doğru modda (ADB/Fastboot) veya Pc de çalışan adb veya fastboot olmadığından emin olun."));
        }
        return this.device.selectConfiguration(match.conf.configurationValue)
            .then(() => this.device.claimInterface(match.intf.interfaceNumber))
            .then(() => this.device.selectAlternateInterface(match.intf.interfaceNumber, match.alt.alternateSetting))
            .then(() => match);
    };
    Adb.WebUSB.Transport.prototype.connectAdb = function(banner, auth_user_notify = null) {
        let VERSION = 0x01000000;
        let VERSION_NO_CHECKSUM = 0x01000001;
        let MAX_PAYLOAD = 256 * 1024;
        let key_idx = 0;
        let AUTH_TOKEN = 1;
        let version_used = Adb.Opt.use_checksum ? VERSION : VERSION_NO_CHECKSUM;
        let m = new Adb.Message("CNXN", version_used, MAX_PAYLOAD, "" + banner + "\0");
        return this.getDevice({ classCode: 255, subclassCode: 66, protocolCode: 1 })
            .then(match => new Adb.WebUSB.Device(this, match))
            .then(adb => m.send_receive(adb)
                .then((function do_auth_response(response) {
                    if (response.cmd != "AUTH" || response.arg0 != AUTH_TOKEN)
                        return response;
                    return keys.then(keys =>
                        do_auth(adb, keys, key_idx++, response.data.buffer, do_auth_response, auth_user_notify));
                }))
                .then(response => {
                    if (response.cmd != "CNXN")
                        throw new Error("Failed to connect with '" + banner + "'");
                    if (response.arg0 != VERSION && response.arg0 != VERSION_NO_CHECKSUM)
                        throw new Error("Version mismatch: " + response.arg0 + " (expected: " + VERSION + " or " + VERSION_NO_CHECKSUM + ")");
                    if (Adb.Opt.debug)
                        console.log("Connected with '" + banner + "', max_payload: " + response.arg1);
                    adb.max_payload = response.arg1;
                    if (response.arg0 == VERSION_NO_CHECKSUM)
                        Adb.Opt.use_checksum = false;
                    adb.banner = new TextDecoder("utf-8").decode(response.data);
                    let pieces = adb.banner.split(':');
                    adb.mode = pieces[0];
                    return adb;
                })
            );
    };
    Adb.WebUSB.Transport.prototype.connectFastboot = function() {
        return this.getDevice({ classCode: 255, subclassCode: 66, protocolCode: 3 })
            .then(match => new Fastboot.WebUSB.Device(this, match))
            .then(fastboot => fastboot.send("getvar:max-download-size")
                .then(() => fastboot.receive()
                    .then(response => {
                        let cmd = decode_cmd(response.getUint32(0, true));
                        if (cmd == "FAIL")
                            throw new Error("Unable to open Fastboot");
                        fastboot.get_cmd = r => decode_cmd(r.getUint32(0, true));
                        fastboot.get_payload = r => r.buffer.slice(4);
                        return fastboot;
                    })
                )
            );
    };
    Adb.WebUSB.Device = function(transport, match) {
        this.transport = transport;
        this.max_payload = 4096;
        this.ep_in = get_ep_num(match.alt.endpoints, "in");
        this.ep_out = get_ep_num(match.alt.endpoints, "out");
    };
    Adb.WebUSB.Device.prototype.open = function(service) {
        return Adb.Stream.open(this, service);
    };
    Adb.WebUSB.Device.prototype.shell = function(command) {
        return Adb.Stream.open(this, "shell:" + command);
    };
    Adb.WebUSB.Device.prototype.reboot = function(command = "") {
        return Adb.Stream.open(this, "reboot:" + command);
    };
    Adb.WebUSB.Device.prototype.send = function(data) {
        if (typeof data === "string") {
            let encoder = new TextEncoder();
            data = encoder.encode(data).buffer;
        }
        if (data != null && data.length > this.max_payload)
            throw new Error("data is too big: " + data.length + " bytes (max: " + this.max_payload + " bytes)");
        return this.transport.send(this.ep_out, data);
    };
    Adb.WebUSB.Device.prototype.receive = function(len) {
        return this.transport.receive(this.ep_in, len);
    };
    Fastboot.WebUSB = {};
    Fastboot.WebUSB.Device = function(transport, match) {
        this.transport = transport;
        this.max_datasize = 64;
        this.ep_in = get_ep_num(match.alt.endpoints, "in");
        this.ep_out = get_ep_num(match.alt.endpoints, "out");
    };
    Fastboot.WebUSB.Device.prototype.send = function(data) {
        if (typeof data === "string") {
            let encoder = new TextEncoder();
            data = encoder.encode(data).buffer;
        }
        if (data != null && data.length > this.max_datasize)
            throw new Error("data is too big: " + data.length + " bytes (max: " + this.max_datasize + " bytes)");
        return this.transport.send(this.ep_out, data);
    };
    Fastboot.WebUSB.Device.prototype.receive = function() {
        return this.transport.receive(this.ep_in, 64);
    };
    Adb.Message = function(cmd, arg0, arg1, data = null) {
        if (cmd.length != 4)
            throw new Error("Invalid command: '" + cmd + "'");
        this.cmd = cmd;
        this.arg0 = arg0;
        this.arg1 = arg1;
        this.length = (data === null) ? 0 : (typeof data === "string") ? data.length : data.byteLength;
        this.data = data;
    };
    Adb.Message.checksum = function(data_view) {
        let sum = 0;
        for (let i = 0; i < data_view.byteLength; i++)
            sum += data_view.getUint8(i);
        return sum & 0xffffffff;
    };
    Adb.Message.send = function(device, message) {
        let header = new ArrayBuffer(24);
        let cmd = encode_cmd(message.cmd);
        let magic = cmd ^ 0xffffffff;
        let data = null;
        let len = 0;
        let checksum = 0;
        if (message.data != null) {
            if (typeof message.data === "string") {
                let encoder = new TextEncoder();
                data = encoder.encode(message.data).buffer;
            } else if (ArrayBuffer.isView(message.data)) {
                data = message.data.buffer;
            } else {
                data = message.data;
            }
            len = data.byteLength;
            if (Adb.Opt.use_checksum)
                checksum = Adb.Message.checksum(new DataView(data));
            if (len > device.max_payload)
                throw new Error("data is too big: " + len + " bytes (max: " + device.max_payload + " bytes)");
        }
        let view = new DataView(header);
        view.setUint32(0, cmd, true);
        view.setUint32(4, message.arg0, true);
        view.setUint32(8, message.arg1, true);
        view.setUint32(12, len, true);
        view.setUint32(16, checksum, true);
        view.setUint32(20, magic, true);
        let seq = device.send(header);
        if (len > 0)
            seq = seq.then(() => device.send(data));
        return seq;
    };
    Adb.Message.receive = function(device) {
        return device.receive(24)
            .then(response => {
                let cmd = response.getUint32(0, true);
                let arg0 = response.getUint32(4, true);
                let arg1 = response.getUint32(8, true);
                let len = response.getUint32(12, true);
                let check = response.getUint32(16, true);
                if (Adb.Opt.use_checksum && response.byteLength > 20) {
                    let magic = response.getUint32(20, true);
                    if ((cmd ^ magic) != -1)
                        throw new Error("magic mismatch");
                }
                cmd = decode_cmd(cmd);
                if (len == 0) {
                    return new Adb.Message(cmd, arg0, arg1);
                }
                return device.receive(len)
                    .then(data => {
                        if (Adb.Opt.use_checksum && Adb.Message.checksum(data) != check)
                            throw new Error("checksum mismatch");
                        return new Adb.Message(cmd, arg0, arg1, data);
                    });
            });
    };
    Adb.Message.prototype.send = function(device) {
        return Adb.Message.send(device, this);
    };
    Adb.Message.prototype.send_receive = function(device) {
        return this.send(device)
            .then(() => Adb.Message.receive(device));
    };
    Adb.Stream = function(device, service, local_id, remote_id) {
        this.device = device;
        this.service = service;
        this.local_id = local_id;
        this.remote_id = remote_id;
    };
    let next_id = 1;
    Adb.Stream.open = function(device, service) {
        let local_id = next_id++;
        let remote_id = 0;
        let m = new Adb.Message("OPEN", local_id, remote_id, "" + service + "\0");
        return m.send_receive(device)
            .then(function do_response(response) {
                if (response.arg1 != local_id)
                    return Adb.Message.receive(device).then(do_response);
                if (response.cmd != "OKAY")
                    throw new Error("Open failed");
                remote_id = response.arg0;
                return new Adb.Stream(device, service, local_id, remote_id);
            });
    };
    Adb.Stream.prototype.close = function() {
        if (this.local_id != 0) {
            this.local_id = 0;
            return this.send("CLSE");
        }
    };
    Adb.Stream.prototype.send = function(cmd, data = null) {
        let m = new Adb.Message(cmd, this.local_id, this.remote_id, data);
        return m.send(this.device);
    };
    Adb.Stream.prototype.receive = function() {
        return Adb.Message.receive(this.device)
            .then(response => {
                if (response.arg0 != 0 && response.arg0 != this.remote_id)
                    throw new Error("Incorrect arg0");
                if (this.local_id != 0 && response.arg1 != this.local_id)
                    throw new Error("Incorrect arg1");
                return response;
            });
    };
    function do_auth(adb, keys, key_idx, token, do_auth_response, auth_user_notify) {
        let AUTH_SIGNATURE = 2;
        let AUTH_RSAPUBLICKEY = 3;
        if (key_idx < keys.length) {
            let slot = keys.length - key_idx - 1;
            let key = keys[slot];
            return crypto.subtle.sign({ name: "RSASSA-PKCS1-v1_5" }, key.privateKey, token)
                .then(signed => {
                    let m = new Adb.Message("AUTH", AUTH_SIGNATURE, 0, signed);
                    return m.send_receive(adb).then(do_auth_response);
                });
        }
        let seq = null;
        let dirty = false;
        if (Adb.Opt.reuse_key !== false) {
            key_idx = Adb.Opt.reuse_key === true ? -1 : Adb.Opt.reuse_key;
            if (key_idx < 0) key_idx += keys.length;
            if (key_idx >= 0 && key_idx < keys.length) {
                seq = Promise.resolve(keys[key_idx]);
            }
        }
        if (seq === null) {
            seq = generate_key();
            dirty = true;
        }
        return seq.then(key => {
            return crypto.subtle.exportKey("spki", key.publicKey)
                .then(pubkey => {
                    let m = new Adb.Message("AUTH", AUTH_RSAPUBLICKEY, 0, toB64(pubkey) + "\0");
                    return m.send(adb);
                })
                .then(() => {
                    if (auth_user_notify != null)
                        auth_user_notify(key.publicKey);
                    return Adb.Message.receive(adb);
                })
                .then(response => {
                    if (response.cmd != "CNXN")
                        return response;
                    if (!dirty)
                        return response;
                    keys.push(key);
                    return db.then(db => store_key(db, key))
                        .then(() => response);
                });
        });
    }
    function promisify(request, onsuccess = "onsuccess", onerror = "onerror") {
        return new Promise(function(resolve, reject) {
            request[onsuccess] = event => resolve(event.target.result);
            request[onerror] = event => reject(event.target.errorCode);
        });
    }
    function init_db() {
        let req = window.indexedDB.open("WebADB", 1);
        req.onupgradeneeded = function(event) {
            let db = event.target.result;
            if (db.objectStoreNames.contains('keys')) {
                db.deleteObjectStore('keys');
            }
            db.createObjectStore("keys", { autoIncrement: true });
        };
        return promisify(req);
    }
    function load_keys(db) {
        let transaction = db.transaction("keys");
        let store = transaction.objectStore("keys");
        let cursor = store.openCursor();
        let keys = [];
        cursor.onsuccess = function(event) {
            let result = event.target.result;
            if (result != null) {
                keys.push(result.value);
                result.continue();
            }
        };
        return promisify(transaction, "oncomplete").then(function(result) {
            return keys;
        });
    }
    function store_key(db, key) {
        let transaction = db.transaction("keys", "readwrite");
        let store = transaction.objectStore('keys');
        let request = store.put(key);
        return promisify(request);
    }
    
    function toB64(buffer) { return btoa(new Uint8Array(buffer).reduce((s, b) => s + String.fromCharCode(b), "")); }
    function get_ep_num(endpoints, dir, type = "bulk") {
        for (let e in endpoints) {
            let ep = endpoints[e];
            if (ep.direction == dir && ep.type == type) {
                return ep.endpointNumber;
            }
        }
        throw new Error("Cannot find " + dir + " endpoint");
    }
    function encode_cmd(cmd) {
        let encoder = new TextEncoder();
        let buffer = encoder.encode(cmd).buffer;
        let view = new DataView(buffer);
        return view.getUint32(0, true);
    }
    function decode_cmd(cmd) {
        let decoder = new TextDecoder();
        let buffer = new ArrayBuffer(4);
        let view = new DataView(buffer);
        view.setUint32(0, cmd, true);
        return decoder.decode(buffer);
    }
    function generate_key() {
        return crypto.subtle.generateKey({
            name: "RSASSA-PKCS1-v1_5",
            modulusLength: Adb.Opt.key_size,
            publicExponent: new Uint8Array([0x01, 0x00, 0x01]),
            hash: { name: "SHA-1" }
        }, Adb.Opt.dump, ["sign", "verify"]);
    }
    
    Adb.Fastboot = Fastboot;
    return Adb;
}));